package db

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/libs/go/ydbtvm"
	"a.yandex-team.ru/security/yadi/web/internal/db/queries"
)

const (
	VersionsDelimiter = ";"
	PypiPath          = "/pypi"
	WebPath           = "/web"
	FeedPath          = "/feed"
)

var ErrNotFound = xerrors.New("record not found")

type (
	DB struct {
		ctx                             context.Context
		sp                              *table.SessionPool
		pypiTablesPath                  string
		webTablesPath                   string
		selectPyPkgVersionQuery         string
		selectPyPkgQuery                string
		listPyPkgsQuery                 string
		selectProjectQuery              string
		selectProjectDependenciesQuery  string
		createProjectQuery              string
		activateProjectQuery            string
		selectNewVulnsQuery             string
		selectChangedVulnsQuery         string
		selectFullFeedQuery             string
		selectOneFeedVulnQueryByYaID    string
		selectOneFeedVulnQueryBySrc     string
		selectOneSrcVulnQuery           string
		selectOneSrcVulnWithActionQuery string
		deleteActionQuery               string
		updateFeedQuery                 string
		changeVulnQuery                 string
	}

	Options struct {
		Database string
		Path     string
		Endpoint string
	}

	FeedLookupOpts struct {
		LastKey uint64
		Limit   uint64
	}

	VulnLookupOpts struct {
		SrcType string
		SrcID   string
		YaID    string
	}
)

func New(ctx context.Context, tvmClient tvm.Client, opts Options) (*DB, error) {
	config := &ydb.DriverConfig{
		Database: opts.Database,
		Credentials: &ydbtvm.TvmCredentials{
			DstID:     ydbtvm.YDBClientID,
			TvmClient: tvmClient,
		},
	}

	driver, err := (&ydb.Dialer{
		DriverConfig: config,
	}).Dial(ctx, opts.Endpoint)

	if err != nil {
		return nil, fmt.Errorf("dial error: %v", err)
	}

	tableClient := table.Client{
		Driver: driver,
	}

	sp := table.SessionPool{
		IdleThreshold: 10 * time.Second,
		Builder:       &tableClient,
	}

	pypiPath := fmt.Sprintf("%s/%s/%s", opts.Database, opts.Path, PypiPath)
	webPath := fmt.Sprintf("%s/%s/%s", opts.Database, opts.Path, WebPath)
	feedPath := fmt.Sprintf("%s/%s/%s", opts.Database, opts.Path, FeedPath)

	err = createTablesWeb(ctx, &sp, webPath)
	if err != nil {
		return nil, fmt.Errorf("create web tables error: %v", err)
	}

	err = createTablesFeed(ctx, &sp, feedPath)
	if err != nil {
		return nil, fmt.Errorf("create feed tables error: %v", err)
	}

	return &DB{
		ctx:            ctx,
		sp:             &sp,
		pypiTablesPath: pypiPath,
		webTablesPath:  webPath,
		// queries for pypi
		selectPyPkgVersionQuery: queries.SelectPyPkgVersionQuery(pypiPath),
		selectPyPkgQuery:        queries.SelectPyPkgQuery(pypiPath),
		listPyPkgsQuery:         queries.ListPyPkgsQuery(pypiPath),
		// queries for projects
		selectProjectQuery:             queries.SelectProjectQuery(webPath),
		selectProjectDependenciesQuery: queries.SelectProjectDependenciesQuery(webPath),
		createProjectQuery:             queries.CreateProjectQuery(webPath),
		activateProjectQuery:           queries.ActivateProjectQuery(webPath),
		// queries for moderate
		selectNewVulnsQuery:             queries.SelectNewVulnsQuery(feedPath),
		selectChangedVulnsQuery:         queries.SelectChangedVulnsQuery(feedPath),
		selectFullFeedQuery:             queries.SelectFullFeedQuery(feedPath),
		selectOneFeedVulnQueryByYaID:    queries.SelectOneFeedVulnQueryByYaID(feedPath),
		selectOneFeedVulnQueryBySrc:     queries.SelectOneFeedVulnQueryBySrc(feedPath),
		selectOneSrcVulnWithActionQuery: queries.SelectOneSrcVulnWithActionQuery(feedPath),
		changeVulnQuery:                 queries.ChangeVulnQuery(feedPath),
		deleteActionQuery:               queries.DeleteActionQuery(feedPath),
	}, nil
}

func (c *DB) Reset() error {
	return c.sp.Close(c.ctx)
}

func createTablesWeb(ctx context.Context, sp *table.SessionPool, prefix string) error {
	err := table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, fmt.Sprintf("%s/%s", prefix, "projects"),
				table.WithColumn("id", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("service", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("name", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("active", ydb.Optional(ydb.TypeBool)),
				table.WithColumn("owners", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("repo_uri", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("severity", ydb.Optional(ydb.TypeFloat)),
				table.WithColumn("skips", ydb.Optional(ydb.TypeJSON)),
				table.WithColumn("targets", ydb.Optional(ydb.TypeJSON)),
				table.WithColumn("dependencies", ydb.Optional(ydb.TypeJSON)),
				table.WithColumn("created_by", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("created_at", ydb.Optional(ydb.TypeInt64)),
				table.WithColumn("updated_at", ydb.Optional(ydb.TypeInt64)),
				table.WithPrimaryKeyColumn("id", "service", "name"),
			)
		}),
	)
	if err != nil {
		return xerrors.Errorf("failed to create projects table: %w", err)
	}

	return nil
}

func createTablesFeed(ctx context.Context, sp *table.SessionPool, prefix string) error {
	err := table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, fmt.Sprintf("%s/%s", prefix, "feed"),
				// PRIMARY KEY
				table.WithColumn("key", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("srcType", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("srcId", ydb.Optional(ydb.TypeUTF8)),

				// tracked fields
				table.WithColumn("cvssScore", ydb.Optional(ydb.TypeFloat)),
				table.WithColumn("vulnVersions", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("title", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("pkgName", ydb.Optional(ydb.TypeUTF8)),
				table.WithColumn("lang", ydb.Optional(ydb.TypeUTF8)),

				table.WithPrimaryKeyColumn("key", "srcType", "srcId"),
			)
		}),
	)
	if err != nil {
		return xerrors.Errorf("failed to create feed table: %w", err)
	}

	return nil
}
