package infra

import (
	"context"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/libs/go/acler"
	"a.yandex-team.ru/security/libs/go/yacsrf"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/manifestor"
	"a.yandex-team.ru/security/yadi/web/internal/config"
	"a.yandex-team.ru/security/yadi/web/internal/db"
	"a.yandex-team.ru/security/yadi/web/internal/vulndb"
	"a.yandex-team.ru/security/yadi/yaudit/pkg/watcher"
)

type Infra struct {
	ctx      context.Context
	CFG      config.Config
	DB       *db.DB
	VulnDB   *vulndb.DB
	TVM      *tvmtool.Client
	BlackBox blackbox.Client
	YaCsrf   *yacsrf.YaCsrf
	ACLer    *acler.ACLer
	Watcher  *watcher.FeedWatcher
}

func New(cfg config.Config) *Infra {
	return &Infra{
		ctx: context.Background(),
		CFG: cfg,
	}
}

func (c *Infra) Start() (err error) {
	c.YaCsrf, err = yacsrf.New(c.CFG.YaCsrfConfig())
	if err != nil {
		return xerrors.Errorf("can't create yacsrf: %w", err)
	}

	c.TVM, err = tvmtool.NewAnyClient()
	if err != nil {
		return xerrors.Errorf("can't create TVM: %w", err)
	}

	c.DB, err = db.New(c.ctx, c.TVM, c.CFG.DBConfig())
	if err != nil {
		return xerrors.Errorf("can't create YDB: %w", err)
	}

	c.VulnDB, err = vulndb.New(c.ctx, c.CFG.VulnDBConfig())
	if err != nil {
		return xerrors.Errorf("can't create VulnDB: %w", err)
	}

	c.ACLer, err = acler.New(acler.User(c.CFG.AdminUsers...), acler.Department(c.CFG.AdminGroups...))
	if err != nil {
		return xerrors.Errorf("can't create ACLer: %w", err)
	}

	c.BlackBox, err = httpbb.NewIntranet(httpbb.WithTVM(c.TVM))
	if err != nil {
		return xerrors.Errorf("can't create blackbox client: %w", err)
	}

	c.Watcher, err = watcher.NewWatcher(
		watcher.WithManifestManager(manifestor.New(c.CFG.ManifestPath.String())),
		watcher.WithFeedToWatch(c.CFG.VulnDBConfig().DatabaseURL),
		watcher.WithCronSchedule(c.CFG.WatcherCronSpec),
		watcher.WithReloadHandler("web", "yadi-web", c.VulnDB.Reload),
		//watcher.WithFailCounter()
	)
	if err != nil {
		return xerrors.Errorf("can't create watcher: %w", err)
	}

	err = c.Watcher.Watch(c.ctx)
	if err != nil {
		return xerrors.Errorf("cant start feed watcher: %w", err)
	}

	return
}

func (c *Infra) Done() (err error) {
	if c.DB != nil {
		err = c.DB.Reset()
		if err != nil {
			return
		}
	}
	return
}
