package server

import (
	"context"
	"fmt"
	"net/http"
	"path"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
)

const (
	runtimePath = "./runtime"
	staticPath  = runtimePath + "/static"
)

type (
	Server struct {
		echo        *echo.Echo
		infra       *infra.Infra
		controllers serverControllers
	}
)

func New(infra *infra.Infra) (*Server, error) {
	return &Server{
		echo:        echo.New(),
		infra:       infra,
		controllers: newControllers(infra),
	}, nil
}

func (s *Server) onStart() (err error) {
	return s.infra.Start()
}

func (s *Server) onEnd() (err error) {
	return s.infra.Done()
}

func (s *Server) ListenAndServe(port uint32) error {
	err := s.onStart()
	if err != nil {
		return xerrors.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := s.onEnd()
		if err != nil {
			simplelog.Error("failed to stop", "err", err)
		}
	}()

	s.echo.HTTPErrorHandler = echoutils.HTTPErrorHandler
	if s.echo.Renderer, err = newTemplateRenderer(); err != nil {
		return err
	}

	// normalize path for old clients
	s.echo.Pre(middleware.RemoveTrailingSlash())

	s.echo.GET("/", func(c echo.Context) error {
		return c.Redirect(http.StatusFound, "/vulns/")
	})
	s.echo.Static("/static", staticPath)
	s.echo.File("/favicon.ico", path.Join(runtimePath, "/static/favicon.ico"))

	s.echo.GET("/ping", func(c echo.Context) error {
		if _, err := s.infra.TVM.GetStatus(c.Request().Context()); err != nil {
			simplelog.Error("failed to ping tvm client", "err", err)
			return c.String(http.StatusInternalServerError, "tvm failed")
		}

		return c.String(http.StatusOK, "pong")
	})

	s.echo.GET("/version", func(c echo.Context) error {
		return c.String(http.StatusOK, buildinfo.Info.ProgramVersion)
	})

	if err = s.controllers.Proxy.BuildRoute(s.echo.Group("/db")); err != nil {
		return err
	}

	if err = s.controllers.Vulndb.BuildRoute(s.echo.Group("/vulns")); err != nil {
		return err
	}

	if err = s.controllers.Confirmator.BuildRoute(s.echo.Group("/project")); err != nil {
		return err
	}

	if err = s.controllers.ModeratePage.BuildRoute(s.echo.Group("/moderate")); err != nil {
		return err
	}

	api := s.echo.Group("/api/v1")

	if err = s.controllers.Python.BuildRoute(api.Group("/python")); err != nil {
		return err
	}

	if err = s.controllers.Common.BuildRoute(api.Group("/common")); err != nil {
		return err
	}

	if err = s.controllers.Releases.BuildRoute(api.Group("/releases")); err != nil {
		return err
	}

	if err = s.controllers.Project.BuildRoute(api.Group("/project")); err != nil {
		return err
	}

	if err = s.controllers.Moderator.BuildRoute(api.Group("/moderate")); err != nil {
		return err
	}

	return s.echo.Start(fmt.Sprintf(":%d", port))
}

func (s *Server) Shutdown(ctx context.Context) error {
	return s.echo.Shutdown(ctx)
}
