package server

import (
	"html/template"
	"io"
	"io/ioutil"
	"path"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	layoutName    = "layout.html"
	templatesPath = runtimePath + "/templates"
)

type (
	TemplateRenderer struct {
		templates map[string]*template.Template
	}
)

var (
	tmplFuncs = template.FuncMap{
		"capitalize": func(str string) string {
			return strings.ToUpper(str[:1]) + str[1:]
		},
		"substr": func(str string, start, end int) string {
			return str[start:end]
		},
		"intRange": func(start, end int) []int {
			n := end - start + 1
			result := make([]int, n)
			for i := 0; i < n; i++ {
				result[i] = start + i
			}
			return result
		},
		"lower": func(str string) string {
			return strings.ToLower(str)
		},
	}
)

func newTemplateRenderer() (*TemplateRenderer, error) {
	tr := &TemplateRenderer{
		templates: make(map[string]*template.Template),
	}

	files, err := ioutil.ReadDir(templatesPath)
	if err != nil {
		return nil, err
	}

	for _, f := range files {
		if f.Name() == layoutName {
			continue
		}

		t, err := template.New(f.Name()).
			Funcs(tmplFuncs).
			ParseFiles(
				path.Join(templatesPath, f.Name()),
				path.Join(templatesPath, layoutName),
			)
		if err != nil {
			return nil, err
		}

		tr.templates[f.Name()] = t
	}
	return tr, nil
}

func (t *TemplateRenderer) Render(w io.Writer, name string, data interface{}, c echo.Context) error {
	tmpl, ok := t.templates[name]
	if !ok {
		return xerrors.Errorf("template %q not found", name)
	}

	return tmpl.ExecuteTemplate(w, layoutName, data)
}
