$(document).ready(function() {
    window.nextPage = null;
    moderation = new Moderation();
    function update () {
        moderation.list().then((vulns) => {
            vulns.forEach(vuln => {
                moderation.addRow(vuln)
            });
        })
    }
    update(); // initial update
    // cleanup #changeSubmit button event listeners and form.change values
    $("#changeModal").on("hidden.bs.modal", function () {
        $("#changeSubmit").unbind();
        $("form.change").trigger("reset");
    })

    $("#nextPageBtn").bind("click", () => {
        update();
    })

    $(document).on("isLastPageEvent", () => {
        $("#nextPageBtn").prop("disabled", true);
    });
});

var Moderation = function() {
    var self = this;
    var api = new API();
    var table = document.querySelector("table.container-fluid"); // fixme: add ID

    // define current page
    let lastPath = window.location.pathname.replace(/\/$/, "").split("/").slice(-1).pop();
    switch (lastPath) {
        case "new":
        self.list = api.getNew;
        self.addRow = addSimpleRow;
        $("#newTab").addClass("active");
        break;

        case "changed":
        self.list = api.getChanged
        self.addRow = addDiffRow
        $("#changedTab").addClass("active");
        break;

        case "feed":
        self.list = api.getAll
        self.addRow = addFeedRow
        $("#feedTab").addClass("active");
        break;

        default:
        self.list = function() {}
        DangerMessage("Bad path. Please, refresh page")
    }

    function addSimpleRow (vuln) {
        var row = table.insertRow(-1);
        row.id = vuln.id;

        // let"s paint this row red
        if (("action" in vuln) && (vuln.action == "delete")) {
            $(row).addClass("delete");
        }
        // Vulnerability cell
        addVulnToCell($(row.insertCell(-1)), vuln, []);

        // Affects cell
        addPkgToCell($(row.insertCell(-1)), vuln, []);

        // Language cell
        $(row.insertCell(-1))
        .text(vuln.language);

        // Feed cell
        $(row.insertCell(-1))
        .text(vuln.src_type);

        // Action buttons cell
        $(row.insertCell(-1))
        .append(approveButton(vuln, {vuln: vuln}, (data) => {
            api.approve(data.data.vuln, true)
        }))
        .append(changeButton(vuln, true))
        .append(deleteButton(vuln));
    }

    function addDiffRow (vuln) {
        var row = table.insertRow(-1);
        row.id = vuln.id;

        // let"s paint this row red
        if (("action" in vuln) && (vuln.action == "delete")) {
            $(row).addClass("delete");
        }

        // Vulnerability cell
        cell = addVulnToCell($(row.insertCell(-1)), vuln, []);
        addVulnToCell(cell, vuln.old, ["mini"]);

         // Affects cell
        cell = addPkgToCell($(row.insertCell(-1)), vuln, []);
        addPkgToCell(cell, vuln.old, ["mini"]);

        // Language cell
        $(row.insertCell(-1))
        .text(vuln.language);

        // Feed cell
        $(row.insertCell(-1))
        .text(vuln.src_type);

        // Action buttons cell
        $(row.insertCell(-1))
        .append(approveButton(vuln, {vuln: vuln}, (data) => {
            api.change(data.data.vuln, true)
        }))
        .append(changeButton(vuln, true))
        .append(deleteButton(vuln));
    }

    function addFeedRow (vuln) {
        var row = table.insertRow(-1);
        row.id = vuln.id;

        // Vulnerability cell
        addVulnToCell($(row.insertCell(-1)), vuln, []);

        // Affects cell
        addPkgToCell($(row.insertCell(-1)), vuln, []);

        // Language cell
        $(row.insertCell(-1))
        .text(vuln.language);

        // Feed cell
        $(row.insertCell(-1))
        .text(vuln.src_type);

        // Action buttons cell
        $(row.insertCell(-1))
        .append(changeButton(vuln, false))
    }

    function addVulnToCell (cell, vuln, classes) {
        var metaclass = classes.join(" ");
        return cell
        .append(
            $("<abbr>", {
                title:  vuln.severity.charAt(0).toUpperCase() + vuln.severity.slice(1),
                text: vuln.cvss_score
            })
            .addClass("label")
            .addClass(`label-${vuln.severity.toLowerCase()}`)
            .addClass(metaclass))
        .append(
            $("<a>", {
                href: vuln.references.pop().url, // here we trust that the original-URL (to feed) will always be the last
                text: vuln.title,
                target: "_blank"})
            .addClass("vuln-title")
            .addClass(metaclass)
            .append("<br>"));
    }

    function addPkgToCell (elm, vuln, classes) {
        var metaclass = classes.join(" ");
        return elm
        .append(
            $("<p>")
            .addClass("affects")
            .append($("<b>").text(vuln.package))
            .append("<br>")
            .append(vuln.vulnerable_versions)
            .addClass(metaclass));
    }

    function approveButton (vuln, event_data, onclick) {
        return $("<button>")
        .attr({
            type: "button",
        })
        .addClass("action-approve btn btn-default btn-success")
        .html(`<span class="glyphicon glyphicon-ok" aria-hidden="true"></span>`)
        .bind("click", event_data, onclick)
    }

    function changeButton (vuln, remove_row) {
        // "change" button
        return $("<button>")
        .attr({
            type: "button",
        })
        .addClass("action-change btn btn-default btn-warning")
        .html(`<span class="glyphicon glyphicon-pencil" aria-hidden="true"></span>`)
        .bind("click", function() {
            // fill form.change fields with current vuln values
            $("form.change").find("#title").attr({
                placeholder: vuln.title,
                value: vuln.title
            });
            $("form.change").find("#cvss_score").attr({
                placeholder: String(vuln.cvss_score),
                value: String(vuln.cvss_score)
            })
            $("form.change").find("#vulnerable_versions").attr({
                placeholder: vuln.vulnerable_versions,
                value: vuln.vulnerable_versions
            })
            $("form.change").find("#package").attr({
                placeholder: vuln.package,
                value: vuln.package
            })
            $("form.change").find(`#language [value="${vuln.language}"]`).attr({
                selected: "selected",
                value: vuln.language
            });
            $("#changeModal").modal("show")

            // sumbmit form button
            $("#changeSubmit").bind("click", function() {
                var newVuln = {
                    src_type: vuln.src_type,
                    id: vuln.id,
                };
                var vulnForm = $("form.change").serializeArray();
                // form values array to Vuln-json
                $.each(vulnForm, function () {
                    if (newVuln[this.name]) {
                        if (!newVuln[this.name].push) {
                            newVuln[this.name] = [newVuln[this.name]];
                        }
                        newVuln[this.name].push(this.value || "");
                    } else {
                        newVuln[this.name] = this.value || "";
                    }
                });

                // cvss-score as float
                try {
                    if (!newVuln["cvss_score"] || !isNaN(parseFloat(newVuln["cvss_score"]))) {
                        newVuln["cvss_score"] = parseFloat(newVuln["cvss_score"])
                    } else {
                        throw new Error(`CVSS score must be a number!`)
                    }
                } catch (error) {
                    DangerMessage(error)
                    return
                };

                // send new vuln
                api.change(newVuln, remove_row);

                $("#changeModal").modal("hide");
            });

        });
    }

    function deleteButton (vuln) {
        return $("<button>")
        .attr({
            type: "button",
        })
        .addClass("action-approve btn btn-default btn-danger")
        .html(`<span class="glyphicon glyphicon-remove" aria-hidden="true"></span>`)
        .bind("click", function() {
            api.delete(vuln, true);
        })
    }
}

var Vuln = function(data) {
    var self = this;
    self.isOld = false;
    if (data == null || typeof(data) === "undefined") {
        throw new Error("Bad vuln");
    }

    self.setOldVersion = function(old) {
        self.old = {};
        self.old = new Vuln(old);
        self.old.isOld = true;
    }

    function parse_vuln(self, vuln) {
        self.id = vuln.id
        self.src_type = vuln.src_type
        self.language = vuln.language
        self.package = vuln.package
        self.cvss_score = vuln.cvss_score
        self.title = vuln.title
        self.vulnerable_versions = vuln.vulnerable_versions
        self.yadi_id = vuln.yadi_id
        self.references = vuln.references
        self.severity = ToSeverity(vuln.cvss_score)
    }

    if (data.hasOwnProperty("action")) {
        self.action = data.action
    } else {
        self.action = null
    }

    if (data.hasOwnProperty("vuln")) {
        parse_vuln(self, data.vuln)
    } else {
        parse_vuln(self, data)
    }

    return self;
}

function ToSeverity(cvss_score) {
    if (typeof(cvss_score) === "number") {
        score = cvss_score
    } else if (typeof(cvss_score) === "string") {
        score = parseFloat(cvss_score)
    } else {
        return "?"
    }

    if (score < 0.1) {
        return "None"
    }
    if (score < 3.9) {
        return "Low"
    }
    if (score < 6.9) {
        return "Medium"
    }
    if (score < 8.9) {
        return "High"
    }

    return "Critical"
}

function DangerMessage(message) {
    $.notify({
        message: message,
    },
    {
        type: "danger",
        delay: 3000,
        mouse_over: "pause",
        z_index: 999999,
    })
}

function SuccessMessage(message) {
    $.notify({
        message: message,
    },
    {
        type: "success",
        delay: 1500,
        mouse_over: "pause",
        z_index: 999999,
    })
}

var API = function() {
    var self = this;
    var path = "/api/v1/moderate/";
    var headers = {
        "Accept": "application/json",
        "Content-Type": "application/json"
    }

    self.sk = function() {
        return get(path + "csrf", "sk").then(sk => {
            return sk
        })
        .catch(err => {
            DangerMessage(err);
        });
    }

    function get(url, result_field) {
        return fetch(url)
        .then(r => {
            if(r.ok) {
                return r.json();
            } else { // fixme: ugly
                throw new Error(`bad response: ${r.status} ${r.statusText}`);
            }
        })
        .then(d => {
            if (d.ok && d.result[result_field]) {
                window.nextPage = d.result.next_page
                if (d.result.is_last) {
                    $(document).trigger("isLastPageEvent");
                }
                return d.result[result_field];
            } else {
                throw new Error("empty result");
            }
        })
    }

    function post (url, data) {
        return self.sk().then(sk => {
            if (sk == "" || typeof(sk) === "undefined") {
                return {ok: false}; // fixme: ugly
            }
            return fetch(path + url, {
                method: "POST",
                headers: {...headers, ...{"X-CSRF-Token": sk}},
                body: data
            }).then(r => {
                if (r.ok) {
                    return r.json();
                } else { // fixme: ugly
                    return r.json().then(err => {
                        if ("error" in err) {
                            throw new Error(`bad response: ${err.error}`);
                        }
                    }).catch(err => {
                        DangerMessage(err);
                        return err;
                    }).then((err) => {
                        if (err == null) {
                            throw new Error(`bad response: ${r.status} ${r.statusText}`);
                        }
                    }).catch(err => {
                        DangerMessage(err);
                        return err;
                    })
                }
            })
        })
    }

    self.approve = function (vuln, remove) {
        post("approve", JSON.stringify(vuln))
        .then(data => {
            if (data.ok) {
                SuccessMessage("Vulnerability approved: " + data.result);
            } else {
                throw new Error(`failed to approve ${data.error}`)
            }
            return vuln.id
        })
        .then(id => {
            if (remove) {
                $(`#${id}`).remove()
            }
        })
        .catch(err => {
            console.log(err);
        })
    }

    self.change = function (vuln, remove) {
        post("change", JSON.stringify(vuln))
        .then(data => {
            if (data.ok) {
                SuccessMessage("Vulnerability changed: " + data.result);
            } else {
                throw new Error(`failed to change ${data.error}`)
            }
            return vuln.id
        })
        .then(id => {
            if (remove) {
                $(`#${id}`).remove()
            }
        })
        .catch(err => {
            console.log(err);
        })
    }

    self.delete = function (vuln, remove) {
        post("reject", JSON.stringify(vuln))
        .then(data => {
            if (data.ok) {
                SuccessMessage("Vulnerability deleted: " + data.result);
            } else {
                throw new Error(`failed to delete ${data.error}`)
            }
            return vuln.id
        })
        .then(id => {
            if (remove) {
                $(`#${id}`).remove()
            }
        })
        .catch(err => {
            console.log(err);
        })
    }

    self.getNew = async function () {
        let vulns = await get(window.nextPage || path + "list/new", "vulns")
        .then(va => {
            var vulns = [];
            va.forEach((vuln) => {
                vulns.push(new Vuln(vuln));
            });
            return vulns;
        })
        return vulns;
    }

    self.getChanged = async function () {
        let vulns = await get(window.nextPage || path + "list/changed", "vulns")
        .then(vd => {
            var vulns = [];
            vd.forEach((diff) => {
                let vuln = new Vuln(diff.new)
                vuln.setOldVersion(diff.old);
                vuln.action = diff.action
                vulns.push(vuln);
            });
            return vulns;
        })
        return vulns;
    }

    self.getAll = async function () {
        let vulns = await get(window.nextPage || path + "list/all", "vulns")
        .then(v => {
            var vulns = [];
            v.forEach((vuln) => {
                vulns.push(new Vuln(vuln));
            });
            return vulns;
        })
        return vulns;
    }
}
