package agroups

import (
	"fmt"
	"path/filepath"
	"strings"
	"time"

	"github.com/BurntSushi/toml"
	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	cacheTTL = 10 * time.Hour
)

var (
	cache = ccache.New(ccache.Configure().MaxSize(512))
)

func ResolveOwners(arcadiaRoot string, owners []string) []string {
	users := make([]string, 0, len(owners))
	usersFromGroup := make([]string, 0, len(owners))
	for _, owner := range owners {
		parts := strings.SplitN(owner, ":", 2)
		if len(parts) == 1 {
			// user
			users = append(users, parts[0])
			continue
		}

		switch parts[0] {
		case "g", "rb":
			// Arcadia group
			resolvedUsers, err := groupUsers(arcadiaRoot, parts[1])
			if err == nil {
				usersFromGroup = append(usersFromGroup, resolvedUsers...)
			}
		default:
			simplelog.Error("failed to resolve project owner", "owner", owner, "err", "unknown owner kind")

		}
	}

	return append(users, usersFromGroup...)
}

func groupUsers(arcadiaRoot string, groupName string) ([]string, error) {
	if c := cache.Get(groupName); c != nil {
		return c.Value().([]string), nil
	}

	groupPath := filepath.Join(arcadiaRoot, "groups", groupName)
	var group Group
	_, err := toml.DecodeFile(groupPath, &group)
	if err != nil {
		return nil, fmt.Errorf("failed to parse Arcadia group: %w", err)
	}

	result := group.Members
	cache.Set(groupName, result, cacheTTL)
	return result, nil
}
