package cli

import (
	"fmt"
	"os"

	"golang.org/x/crypto/ssh/terminal"

	"a.yandex-team.ru/security/yadi/yadi-arc/internal/formatter"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/formatter/consolefmt"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/formatter/jsonfmt"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/formatter/textfmt"
)

const (
	TextFormatterName    = "text"
	ConsoleFormatterName = "console"
	JSONFormatterName    = "json"
)

func AnalyzeFormatter(format string) (formatterFn formatter.AnalyzeFn, err error) {
	switch format {
	case TextFormatterName:
		formatterFn = textfmt.IssuesFormatter
	case ConsoleFormatterName:
		formatterFn = consolefmt.IssuesFormatter
	case JSONFormatterName:
		formatterFn = jsonfmt.AnalyzeFormatter
	default:
		err = fmt.Errorf(
			"unrecognized format %q, must be one of: text, console or json",
			format,
		)
	}

	return
}

func ListFormatter(format string) (formatterFn formatter.ListFn, err error) {
	switch format {
	case TextFormatterName:
		formatterFn = textfmt.ListFormatter
	case ConsoleFormatterName:
		formatterFn = consolefmt.ListFormatter
	case JSONFormatterName:
		formatterFn = jsonfmt.ListFormatter
	default:
		err = fmt.Errorf(
			"unrecognized format %q, must be one of: text, console or json",
			format,
		)
	}
	return
}

func DetectDefaultFormatter() string {
	switch {
	case terminal.IsTerminal(int(os.Stdout.Fd())):
		return ConsoleFormatterName
	default:
		return TextFormatterName
	}
}
