package commands

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/config"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/formatter/policyfmt"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze/collector/contrib"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze/collector/policy"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze/collector/project"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/yamake"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

func newAnalyzer(opts ...analyze.Option) (*analyze.Analyzer, error) {
	analyzeOpts := []analyze.Option{
		analyze.WithFeedOptions(feed.Options{
			MinimumSeverity: config.MinimumSeverity,
			FeedURI:         config.FeedURI,
		}),
		analyze.WithExpandOwners(rootOpts.ExpandOwners),
	}

	return analyze.New(
		rootOpts.ArcadiaRoot,
		append(analyzeOpts, opts...)...,
	)
}

func doList(pm manager.Manager, opts ...analyze.Option) error {
	analyzer, err := newAnalyzer(opts...)
	if err != nil {
		return fmt.Errorf("failed to create analyzer: %w", err)
	}

	listResults, err := analyzer.List(context.Background(), pm)
	if err != nil {
		return err
	}

	format, err := cli.ListFormatter(rootOpts.Format)
	if err != nil {
		return err
	}

	fmt.Println(format(listResults))
	return nil
}

func doAnalyzeProject(pm manager.Manager, opts ...analyze.Option) error {
	opts = append(opts, analyze.WithCollectorProvider(project.NewProvider()))
	return doAnalyze(pm, opts...)
}

func doAnalyzeContib(pm manager.Manager, opts ...analyze.Option) error {
	opts = append(opts, analyze.WithCollectorProvider(contrib.NewProvider()), analyze.WithVulnerabilityExcludes(rootOpts.SkipIssues), analyze.WithVulnerabilityWhitelist(rootOpts.OnlyIssues))
	return doAnalyze(pm, opts...)
}

func doAnalyze(pm manager.Manager, opts ...analyze.Option) error {
	analyzer, err := newAnalyzer(opts...)
	if err != nil {
		return fmt.Errorf("failed to create analyzer: %w", err)
	}

	analyzeResults, err := analyzer.Analyze(context.Background(), pm)
	if err != nil {
		return err
	}

	if len(analyzeResults) == 0 {
		switch rootOpts.Format {
		case cli.JSONFormatterName:
			fmt.Println("[]")
		default:
			fmt.Println("No issues found.")
		}
		return nil
	}

	format, err := cli.AnalyzeFormatter(rootOpts.Format)
	if err != nil {
		return err
	}

	fmt.Println(format(analyzeResults))
	os.Exit(rootOpts.ExitStatus)
	return nil
}

func doGeneratePolicy(pm manager.Manager, opts ...analyze.Option) error {
	config.MinimumSeverity = cvs.HighSeverity // TODO(melkikh): are you sure?
	formatFn := policyfmt.PolicyFormatterWithExcludes()
	opts = append(opts, analyze.WithCollectorProvider(policy.NewProvider(rootOpts.ArcadiaRoot, nil)), analyze.WithVulnerabilityWhitelist(rootOpts.OnlyIssues))
	analyzer, err := newAnalyzer(opts...)
	if err != nil {
		return fmt.Errorf("failed to create analyzer: %w", err)
	}

	analyzeResults, err := analyzer.Analyze(context.Background(), pm)
	if err != nil {
		return err
	}

	fmt.Println(formatFn(analyzeResults))
	return nil
}

func readFixExcludes(path string) (policyfmt.Excludes, error) {
	var affectedTargets map[string][]string

	data, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, fmt.Errorf("failed to read file with excludes: %w", err)
	}

	if err = json.Unmarshal(data, &affectedTargets); err != nil {
		return nil, fmt.Errorf("failed to unmarshal file with excludes: %w", err)
	}

	result := make(policyfmt.Excludes)
	for platform, issues := range affectedTargets {
		for _, issue := range issues {
			if _, ok := result[issue]; ok {
				simplelog.Info("duplicate for exclude path", "path", issue, "platform", platform)
				continue
			}
			result[issue] = struct{}{}
		}
	}
	return result, nil
}

func doFixPolicy(_ manager.Manager, excludes policyfmt.Excludes, opts ...analyze.Option) error {
	config.MinimumSeverity = cvs.HighSeverity // TODO(melkikh): are you sure?

	var jTargets []yamake.YMakeTarget
	for excludeSrc := range excludes {
		jTargets = append(jTargets, yamake.YMakeTarget{
			Path:      filepath.Join(rootOpts.ArcadiaRoot, excludeSrc, "ya.make"),
			ModuleDir: excludeSrc,
			Type:      yamake.TargetTypeJava,
		})
	}

	pm, err := yamake.NewManager(jTargets, yamake.ManagerOpts{
		ArcadiaPath: rootOpts.ArcadiaRoot,
		BatchSize:   testOpts.BatchSize, //TODO(melkikh): move me
	})
	if err != nil {
		return err
	}

	formatFn := policyfmt.PolicyFormatterWithExcludes()
	opts = append(opts, analyze.WithCollectorProvider(policy.NewProvider(rootOpts.ArcadiaRoot, excludes)))
	analyzer, err := newAnalyzer(opts...)
	if err != nil {
		return fmt.Errorf("failed to create analyzer: %w", err)
	}

	analyzeResults, err := analyzer.Analyze(context.Background(), pm)
	if err != nil {
		return err
	}

	fmt.Println(formatFn(analyzeResults))
	return nil
}
