package commands

import (
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"strings"

	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/config"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
)

var (
	rootOpts struct {
		ArcadiaRoot         string
		ExpandOwners        bool
		SeverityLevel       int
		Format              string
		FeedURI             string
		RawSkipIssues       string
		SkipIssues          []string
		RawOnlyIssues       string
		OnlyIssues          []string
		CheckGlobalExcludes bool
		Verbose             bool
		ShowVersion         bool
		ExitStatus          int
		ListOnly            bool
	}
)

func Execute() error {
	rootCmd.Version = config.FullVersion()
	rootCmd.SetVersionTemplate("{{.Version}}\n")

	return rootCmd.Execute()
}

var rootCmd = &cobra.Command{
	Use:          "yadi-arc",
	SilenceUsage: true,
	Short:        "Yandex Dependency Inspector for Arcadia",
	Long: `Yandex Dependency Inspector for Arcadia

Documentation: https://wiki.yandex-team.ru/product-security/yadi-arc/
Vulnerability Database: https://yadi.yandex-team.ru/vulns`,
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		var root *cobra.Command
		for p := cmd; p != nil; p = p.Parent() {
			if p.Name() == "yadi-arc" {
				root = p
				break
			}
		}

		if root == nil {
			return errors.New("can't find root command ('yadi-arc')")
		}

		if err := parseOpts(); err != nil {
			return err
		}

		if rootOpts.ShowVersion {
			fmt.Println(config.FullVersion())
			return nil
		}

		if err := flagsToConfig(); err != nil {
			return err
		}

		if rootOpts.Verbose {
			simplelog.SetLevel(simplelog.DebugLevel)
		} else {
			simplelog.SetLevel(simplelog.InfoLevel)
		}

		target := "."
		if len(args) > 0 {
			target = args[0]
		}
		arcadiaRoot, err := ya.FindArcadiaRoot(target)
		if err != nil {
			return fmt.Errorf("failed to find arcadia root: %w", err)
		}

		ya.SetArcadiaRoot(arcadiaRoot)
		rootOpts.ArcadiaRoot = arcadiaRoot

		if rootOpts.CheckGlobalExcludes {
			if err := parseExcludesFile(arcadiaRoot); err != nil {
				return fmt.Errorf("failed to parse excludes file: %w", err)
			}
		}
		return nil
	},
}

func init() {
	flags := rootCmd.PersistentFlags()
	flags.BoolVar(&rootOpts.ListOnly, "list", false,
		"list parsed dependency and exit")
	flags.StringVarP(&rootOpts.Format, "format", "f", cli.DetectDefaultFormatter(),
		"output format (text, console or json)")
	flags.CountVarP(&rootOpts.SeverityLevel, "level", "l",
		"report issues of a given severity level or higher (-l for Low, -ll for Medium, -lll for High, -llll Critical) (default: Medium)")
	flags.StringVarP(&rootOpts.RawSkipIssues, "skips", "s", "",
		"comma separated list of skipped issues")
	flags.BoolVar(&rootOpts.CheckGlobalExcludes, "global-excludes", false, "use arcadia-wide excludes file")
	flags.StringVarP(&rootOpts.RawOnlyIssues, "only", "i", "",
		"comma separated list of whitelisted issues")
	flags.BoolVar(&rootOpts.ExpandOwners, "expand-groups", false,
		"expand owner groups")
	flags.StringVar(&rootOpts.FeedURI, "feed", "",
		"vulnerability DB URI")
	flags.BoolVar(&rootOpts.Verbose, "verbose", false,
		"verbose output")
	flags.IntVar(&rootOpts.ExitStatus, "exit-status", 3,
		"set specified exit status when something found")
	flags.BoolVar(&rootOpts.ShowVersion, "version", false,
		"show the current version")

	rootCmd.AddCommand(
		versionCmd,
		testCmd,
		contribCmd,
		policyCmd,
	)
}

func flagsToConfig() error {
	score, err := cvs.FromLevel(rootOpts.SeverityLevel)
	if err != nil {
		return err
	}

	config.SkipIssues = rootOpts.SkipIssues
	config.OnlyIssues = rootOpts.OnlyIssues
	config.MinimumSeverity = score
	if rootOpts.FeedURI != "" {
		config.FeedURI = rootOpts.FeedURI
	}

	if _, err := cli.ListFormatter(rootOpts.Format); err != nil {
		return err
	}

	return nil
}

func parseOpts() error {
	// Post process skips
	if rootOpts.RawSkipIssues != "" {
		skips := strings.Split(rootOpts.RawSkipIssues, ",")
		for _, skip := range skips {
			rootOpts.SkipIssues = append(rootOpts.SkipIssues, strings.TrimSpace(skip))
		}
	}

	if rootOpts.RawOnlyIssues != "" {
		whitelist := strings.Split(rootOpts.RawOnlyIssues, ",")
		for _, only := range whitelist {
			rootOpts.OnlyIssues = append(rootOpts.OnlyIssues, strings.TrimSpace(only))
		}
	}

	return nil
}

func parseExcludesFile(arcadiaRoot string) error {
	const excludesFile = "security/yadi/rules/excludes.yaml"
	excludesFilePath := path.Join(arcadiaRoot, excludesFile)
	excludesData, err := ioutil.ReadFile(excludesFilePath)
	switch {
	case os.IsNotExist(err):
		simplelog.Warn("arcadia-wide excludes file not found", "file", excludesFilePath, "err", err)
		return nil
	case err != nil:
		return err
	}

	excludes := make([]string, 0)

	if err = yaml.Unmarshal(excludesData, &excludes); err != nil {
		return err
	}

	for _, skip := range excludes {
		rootOpts.SkipIssues = append(rootOpts.SkipIssues, strings.TrimSpace(skip))
	}

	return nil
}
