package textfmt

import (
	"bytes"
	"fmt"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
)

const tmpl = `
{{range $i, $module := . }}
[{{ $i }}] {{ $module.Path }}{{if $module.Owners}} (owners: {{ join $module.Owners ","}}){{end}}:
{{range $_, $issue := $module.Issues}}
  {{ $issue.Severity }} severity vulnerability found on {{ $issue.String }}:
  - desc: {{ $issue.Summary }}
  - id: {{ $issue.ID }}
  - info: {{ $issue.Reference }}
{{- if $issue.LocalPath}}
  - path: {{ $issue.LocalPath }}
{{- end}}
  - vulnerable versions: {{ $issue.RawVersions }}
{{- if $issue.VulnPaths }}
  - vulnerable paths: {{ $issue.VulnPaths }}
  - vulnerable path samples:
{{- range $_, $paths := $issue.VulnPathSamples }}
    * {{join $paths " > " }}
{{- end -}}
{{ end }}
{{ end }}
{{- end -}}
`

var templateFns = template.FuncMap{
	"join": strings.Join,
}

func IssuesFormatter(results analyze.ResultAnalyze) string {
	t := template.Must(template.New("text_result").Funcs(templateFns).Parse(tmpl))
	var result bytes.Buffer

	err := t.Execute(&result, results)
	if err != nil {
		simplelog.Error("failed to render results", "err", err)
	}

	return result.String()
}

func ListFormatter(results analyze.ResultList) string {
	result := strings.Builder{}

	for _, module := range results {
		result.WriteString(fmt.Sprintf("%s:\n", module.LocalPath))
		result.WriteString(fmt.Sprintf("  - package name: %s\n", module.Name))
		result.WriteString(fmt.Sprintf("  - version: %s\n", module.Version))
		if len(module.Owners) > 0 {
			result.WriteString(fmt.Sprintf("  - owners: %s\n", strings.Join(module.Owners, ",")))
		}

		if len(module.Dependencies) > 0 {
			result.WriteString("  - dependencies:\n")
			for _, dep := range module.Dependencies {
				printDep(&result, dep, 1)
			}
		}
		result.WriteByte('\n')
	}
	return result.String()
}

func printDep(result *strings.Builder, dep manager.Module, indent int) {
	prefix := strings.Repeat("  ", indent)
	result.WriteString(fmt.Sprintf("  %s* %s\n", prefix, dep.String()))
	if len(dep.Dependencies) != 0 {
		for _, subDep := range dep.Dependencies {
			printDep(result, subDep, indent+1)
		}
	}
}
