package ya

import (
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"sync"

	lru "github.com/hashicorp/golang-lru"

	"a.yandex-team.ru/library/go/yatool"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yamake"
)

const (
	YaMakeFilename = "ya.make"
)

var (
	arcadiaRoot = "."
	pathOnce    sync.Once
	yaPath      string
	yaCache     = func() *lru.Cache {
		c, err := lru.New(2048)
		if err != nil {
			panic(fmt.Sprintf("can't create ya.make cache: %s", err.Error()))
		}
		return c
	}()
)

func SetArcadiaRoot(root string) {
	if arcadiaRoot != "." {
		panic("arcadia root is already seted")
	}
	arcadiaRoot = root
}

func Run(args []string, env []string, dir string) (result Output, err error) {
	simplelog.Info("run ya", "args", strings.Join(args, " "))
	yaCmd := exec.Command(Path(), args...)
	yaCmd.Dir = dir
	yaCmd.Env = append(os.Environ(), env...)
	yaCmd.Stdout = &result.stdout
	yaCmd.Stderr = &result.stderr

	if err = yaCmd.Run(); err != nil {
		if exitErr, ok := err.(*exec.ExitError); ok {
			result.exitCode = exitErr.ExitCode()
		}
		return result, fmt.Errorf("failed to run ya: %w", err)
	}
	return
}

func Path() string {
	pathOnce.Do(func() {
		var err error
		yaPath, err = yatool.FindYa(arcadiaRoot)
		if err != nil {
			simplelog.Warn("failed to find ya binary, system ya used", "err", err)
			yaPath = "ya"
		}
	})

	return yaPath
}

func ParseYaMake(yaMakePath string) (*yamake.YMake, error) {
	if cached, ok := yaCache.Get(yaMakePath); ok {
		return cached.(*yamake.YMake), nil
	}

	target, err := yamake.ParseFile(yaMakePath)
	if err != nil {
		return nil, err
	}

	_ = yaCache.Add(yaMakePath, target)
	return target, nil
}

func ModuleOwners(modulePath string) []string {
	target, err := ParseYaMake(filepath.Join(modulePath, YaMakeFilename))
	if err != nil {
		// that's fine
		simplelog.Debug("unable to parse ya.make", "module_path", modulePath, "err", err)
		return nil
	}

	return target.Owners
}

func ModuleVersion(modulePath string) string {
	target, err := ParseYaMake(filepath.Join(modulePath, YaMakeFilename))
	if err != nil {
		// that's fine
		simplelog.Debug("unable to parse ya.make", "module_path", modulePath, "err", err)
		return ""
	}

	ver := target.Version
	return ver
}
