package analyze

import (
	"context"
	"errors"
	"fmt"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/agroups"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

type (
	Analyzer struct {
		feedFetcher    *feed.Fetcher
		feeds          map[string]feed.Vulnerabilities
		arcadiaRoot    string
		resolveGroups  bool
		issueCollector CollectorProvider
		excludedVulns  map[string]struct{}
		vulnsWhitelist map[string]struct{}
	}
)

var (
	ErrCanceled = errors.New("canceled")
)

func New(arcadiaRoot string, opts ...Option) (*Analyzer, error) {
	analyzer := &Analyzer{
		arcadiaRoot:   arcadiaRoot,
		resolveGroups: false,
		feedFetcher: feed.New(feed.Options{
			MinimumSeverity: cvs.MediumSeverity,
			FeedURI:         feed.DefaultURI,
		}),
		feeds: make(map[string]feed.Vulnerabilities),
	}

	for _, opt := range opts {
		opt(analyzer)
	}

	return analyzer, nil
}

func (a *Analyzer) Analyze(ctx context.Context, pm manager.Manager) (ResultAnalyze, error) {
	simplelog.Info("resolving modules")
	if a.issueCollector == nil {
		return nil, errors.New("no issue collector provided")
	}

	collector := a.issueCollector()

	for pm.NextModule() {
		module := pm.Module()

		vulnFeed, err := a.fetchFeed(ctx, module.Language)
		if err != nil {
			simplelog.Error("failed to fetch vulnFeed, ignore root module",
				"module", module.Name,
				"local_path", module.LocalPath,
				"err", err,
			)
			continue
		}

		walker := modulesWalker{
			rootModule:     module,
			excludedVulns:  a.excludedVulns,
			vulnsWhitelist: a.vulnsWhitelist,
			visitedModules: make(map[string]struct{}),
			collector:      collector,
			vulnFeed:       vulnFeed,
		}

		simplelog.Info("analyzing modules tree", "module", module.Name, "local_path", module.LocalPath)
		err = walker.Walk(ctx)
		if err != nil {
			simplelog.Error("failed to analyze module tree", "module", module.Name, "local_path", module.LocalPath, "err", err)
			continue
		}
	}

	result := collector.Results()
	if a.resolveGroups {
		for i := range result {
			result[i].Owners = agroups.ResolveOwners(a.arcadiaRoot, result[i].Owners)
		}
	}

	simplelog.Info("Done")
	return result, pm.Err()
}

func (a *Analyzer) List(_ context.Context, pm manager.Manager) (ResultList, error) {
	simplelog.Info("resolving modules")

	var result ResultList
	for pm.NextModule() {
		result = append(result, pm.Module())
	}

	simplelog.Info("Done")
	return result, pm.Err()
}

func (a *Analyzer) fetchFeed(_ context.Context, language string) (feed.Vulnerabilities, error) {
	if f, ok := a.feeds[language]; ok {
		return f, nil
	}

	f, err := a.feedFetcher.Fetch(language)
	if err != nil {
		// * * WARNING! * *
		// TODO: Remove after cpp NVD feed release!
		if language == "cpp" {
			simplelog.Debug("No C++ feed found!")
			f = feed.Vulnerabilities{} // SUPPRESS ERRORS OF CPP FEED NOT FOUND!
		} else {
			return nil, fmt.Errorf("failed to fetch vulnFeed for language '%s': %w", language, err)
		}
		// * * * * * * * * *
	}

	a.feeds[language] = f
	return f, nil
}
