package contrib

import (
	"sort"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

const (
	maxPaths = 5
)

var _ analyze.Collector = (*IssueCollector)(nil)

type cModule struct {
	path    string
	owners  []string
	version string
	issues  map[string]*analyze.Issue
}

type IssueCollector struct {
	modules map[string]*cModule
}

func NewProvider() analyze.CollectorProvider {
	return func() analyze.Collector {
		return NewIssueCollector()
	}
}

func NewIssueCollector() *IssueCollector {
	return &IssueCollector{
		modules: make(map[string]*cModule),
	}
}

func (c *IssueCollector) Collect(vuln feed.Vulnerability, module manager.Module, vulnPath []manager.Module) {
	if len(vulnPath) == 0 {
		simplelog.Error("failed to collect issues", "module", module.Name, "err", "no vuln path")
		return
	}

	if m, ok := c.modules[module.LocalPath]; ok {
		if issue, ok := m.issues[vuln.ID]; ok {
			issue.VulnPaths++
			if issue.VulnPaths >= maxPaths {
				return
			}

			found := false
			rootModule := vulnPath[0].Name
			for _, p := range issue.VulnPathSamples {
				if p[0] == rootModule {
					found = true
					break
				}
			}

			if !found {
				issue.VulnPathSamples = append(issue.VulnPathSamples, manager.BuildPath(vulnPath))
			}
			return
		}

		m.issues[vuln.ID] = &analyze.Issue{
			Vulnerability:   vuln,
			Version:         module.Version.String(),
			VulnPaths:       1,
			VulnPathSamples: [][]string{manager.BuildPath(vulnPath)},
		}

		return
	}

	c.modules[module.LocalPath] = &cModule{
		owners:  module.Owners,
		path:    module.LocalPath,
		version: module.Version.String(),
		issues: map[string]*analyze.Issue{
			vuln.ID: {
				Vulnerability:   vuln,
				Version:         module.Version.String(),
				VulnPaths:       1,
				VulnPathSamples: [][]string{manager.BuildPath(vulnPath)},
			},
		},
	}
}

func (c *IssueCollector) Results() []analyze.VulnerableModule {
	out := make([]analyze.VulnerableModule, len(c.modules))
	i := 0
	for _, module := range c.modules {
		vulnerableModule := analyze.VulnerableModule{
			Path:    module.path,
			Owners:  module.owners,
			Version: module.version,
			Issues:  make(analyze.IssueList, len(module.issues)),
		}

		k := 0
		for _, issue := range module.issues {
			sort.Slice(issue.VulnPathSamples, func(i, j int) bool {
				return len(issue.VulnPathSamples[i]) < len(issue.VulnPathSamples[j])
			})

			vulnerableModule.Issues[k] = *issue
			k++
		}

		sort.Sort(vulnerableModule.Issues)
		out[i] = vulnerableModule
		i++
	}

	sort.Slice(out, func(i, j int) bool {
		return out[i].Path < out[j].Path
	})
	return out
}
