package policy

import (
	"sort"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

var _ analyze.Collector = (*IssueCollector)(nil)

type cModule struct {
	path      string
	owners    []string
	version   string
	issues    map[string]*analyze.Issue
	consumers map[string]struct{}
}

type IssueCollector struct {
	arcadiaRootPath string
	modules         map[string]*cModule
	excludes        map[string]struct{}
}

func NewProvider(arcadiaRootPath string, excludes map[string]struct{}) analyze.CollectorProvider {
	return func() analyze.Collector {
		collector := NewIssueCollector()
		collector.arcadiaRootPath = arcadiaRootPath
		collector.excludes = excludes
		return collector
	}
}

func NewIssueCollector() *IssueCollector {
	return &IssueCollector{
		modules: make(map[string]*cModule),
	}
}

func (c *IssueCollector) Collect(vuln feed.Vulnerability, module manager.Module, vulnPath []manager.Module) {
	if len(vulnPath) == 0 {
		simplelog.Error("failed to collect issues", "module", module.Name, "err", "no vuln path")
		return
	}

	if m, ok := c.modules[module.LocalPath]; ok {
		if issue, ok := m.issues[vuln.ID]; ok {
			issue.VulnPaths++
			return
		}

		m.issues[vuln.ID] = &analyze.Issue{
			Vulnerability:   vuln,
			Version:         module.Version.String(),
			VulnPaths:       1,
			VulnPathSamples: nil,
		}

		consumerPath, skip := getConsumer(vulnPath)
		if skip {
			return
		}

		consumer := strings.TrimPrefix(consumerPath, c.arcadiaRootPath)
		if _, ok := m.consumers[consumer]; ok {
			return
		}

		m.consumers[consumer] = struct{}{}
		return
	}

	m := &cModule{
		owners:  module.Owners,
		path:    module.LocalPath,
		version: module.Version.String(),
		issues: map[string]*analyze.Issue{
			vuln.ID: {
				Vulnerability:   vuln,
				Version:         module.Version.String(),
				VulnPaths:       1,
				VulnPathSamples: nil,
			},
		},
		consumers: make(map[string]struct{}),
	}

	consumerPath, skip := getConsumer(vulnPath)
	if !skip {
		consumer := strings.TrimPrefix(consumerPath, c.arcadiaRootPath)
		m.consumers[consumer] = struct{}{}
	}

	c.modules[module.LocalPath] = m
}

func (c *IssueCollector) Results() []analyze.VulnerableModule {
	out := make([]analyze.VulnerableModule, len(c.modules))
	i := 0
	for _, module := range c.modules {
		vulnerableModule := analyze.VulnerableModule{
			Path:      strings.TrimPrefix(module.path, c.arcadiaRootPath+"/"),
			Owners:    module.owners,
			Version:   module.version,
			Issues:    make(analyze.IssueList, len(module.issues)),
			Consumers: make(analyze.ConsumerList, len(module.consumers)),
		}

		j := 0
		for _, issue := range module.issues {
			vulnerableModule.Issues[j] = *issue
			j++
		}
		sort.Sort(vulnerableModule.Issues)

		k := 0
		for consumer := range module.consumers {
			vulnerableModule.Consumers[k] = consumer
			k++
		}

		sort.Slice(vulnerableModule.Consumers, func(i, j int) bool {
			return vulnerableModule.Consumers[i] < vulnerableModule.Consumers[j]
		})

		out[i] = vulnerableModule
		i++
	}

	sort.Slice(out, func(i, j int) bool {
		return out[i].Path < out[j].Path
	})
	return out
}

func getConsumer(path []manager.Module) (string, bool) {
	if len(path) <= 1 {
		return "", true
	}
	return path[len(path)-2].LocalPath, false
}
