package project

import (
	"sort"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

var _ analyze.Collector = (*IssueCollector)(nil)

type cModule struct {
	path   string
	owners []string
	issues map[string]*analyze.Issue
}

type IssueCollector struct {
	modules map[string]*cModule
}

func NewProvider() analyze.CollectorProvider {
	return func() analyze.Collector {
		return NewIssueCollector()
	}
}

func NewIssueCollector() *IssueCollector {
	return &IssueCollector{
		modules: make(map[string]*cModule),
	}
}

func (c *IssueCollector) Collect(vuln feed.Vulnerability, module manager.Module, vulnPath []manager.Module) {
	if len(vulnPath) == 0 {
		simplelog.Error("failed to collect issues", "module", module.Name, "err", "no vuln path")
		return
	}

	issueKey := module.LocalPath + vuln.ID
	rootModule := vulnPath[0]
	if m, ok := c.modules[rootModule.LocalPath]; ok {
		if _, ok := m.issues[issueKey]; ok {
			return
		}

		m.issues[issueKey] = &analyze.Issue{
			Vulnerability: vuln,
			Version:       module.Version.String(),
			LocalPath:     module.LocalPath,
		}
		return
	}

	// vulnerable path useless for project results
	c.modules[rootModule.LocalPath] = &cModule{
		owners: rootModule.Owners,
		path:   rootModule.LocalPath,
		issues: map[string]*analyze.Issue{
			issueKey: {
				Vulnerability: vuln,
				Version:       module.Version.String(),
				LocalPath:     module.LocalPath,
			},
		},
	}
}

func (c *IssueCollector) Results() []analyze.VulnerableModule {
	out := make([]analyze.VulnerableModule, len(c.modules))
	i := 0
	for _, module := range c.modules {
		vulnerableModule := analyze.VulnerableModule{
			Path:   module.path,
			Owners: module.owners,
			Issues: make(analyze.IssueList, len(module.issues)),
		}

		k := 0
		for _, issue := range module.issues {
			vulnerableModule.Issues[k] = *issue
			k++
		}

		sort.Sort(vulnerableModule.Issues)
		out[i] = vulnerableModule
		i++
	}

	sort.Slice(out, func(i, j int) bool {
		return out[i].Path < out[j].Path
	})
	return out
}
