package cpp

import (
	"errors"
	"path"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
)

const (
	arcadiaContribPrefix = "contrib/libs"
	language             = "cpp"
	name                 = "arcadia-cpp"
)

var errAlreadyProcessed = errors.New("package had already been processed")

var _ manager.Manager = (*Manager)(nil)

type (
	ManagerOpts struct {
		ArcadiaPath string // Absolute path
	}

	set     map[string]struct{}
	Manager struct {
		arcadiaRoot   string
		targets       []string
		targetPos     int
		currentModule manager.Module
		lastError     error
		processed     set
	}
)

func (s set) Contains(pkg string) bool {
	_, ok := s[pkg]
	return ok
}

func (s set) Add(pkg string) {
	s[pkg] = struct{}{}
}

func NewManager(targets []string, opts ManagerOpts) (*Manager, error) {
	return &Manager{
		arcadiaRoot: opts.ArcadiaPath,
		targets:     targets,
		targetPos:   -1,
		processed:   make(set),
	}, nil
}

func (m Manager) Name() string {
	return name
}

// Process next module if we can
// Failed entries skipped
func (m *Manager) NextModule() bool {
	m.targetPos++
	if m.targetPos >= len(m.targets) {
		return false
	}

	requiredPkg := m.targets[m.targetPos]
	module, err := m.createModule(requiredPkg)
	if err != nil {
		if err != errAlreadyProcessed {
			m.lastError = err
			simplelog.Error("failed to resolve pkg", "pkg", requiredPkg, "err", err)
		}
		return m.NextModule()
	}
	m.currentModule = module

	return true
}

func (m *Manager) Err() error {
	return m.lastError
}

func (m *Manager) Module() manager.Module {
	return m.currentModule
}

func (m *Manager) createModule(pkgName string) (manager.Module, error) {
	localPath := pkgName
	pkgFilepath := m.resolvePath(pkgName)
	ver := ya.ModuleVersion(pkgFilepath)

	cppVersion, _ := versionarium.NewVersion("cpp", ver) // error is always nil
	resolvedPkgName := resolvePackageName(pkgFilepath)
	if m.processed.Contains(resolvedPkgName) {
		return manager.Module{}, errAlreadyProcessed
	} else {
		m.processed.Add(resolvedPkgName)
	}
	return manager.Module{
		Name:      resolvedPkgName,
		Language:  language,
		Version:   cppVersion,
		LocalPath: localPath,
	}, nil
}

// Сlean path and concat with arcadia root
func (m *Manager) resolvePath(filename string) string {
	filename = path.Clean(filename)
	if path.IsAbs(filename) {
		return filename
	}
	return path.Join(m.arcadiaRoot, filename)
}

// Given path resolves package name
// Example:
// resolvePackageName("/some/path/with/lib/libOpenSSL/ya.make") -> "openssl"
// resolvePackageName("/some/path/with/lib/libOpenSSL") -> "openssl"
func resolvePackageName(filepath string) string {
	if path.Base(filepath) == ya.YaMakeFilename {
		baseLen := len(path.Base(filepath))
		filepath = filepath[:len(filepath)-baseLen]
	}
	return versionarium.Canonize(path.Base(filepath))
}
