package cpp

import (
	"errors"
	"fmt"
	"io/fs"
	"os"
	"path/filepath"
	"reflect"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yamake"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/blacklist"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
)

var errNotCppModule = errors.New("not cpp module")
var noVersions = make(set)

func FindTargets(arcadiaRoot string) ([]string, error) {
	contribRoot := filepath.Join(arcadiaRoot, arcadiaContribPrefix)
	if !fsutils.IsDirExists(contribRoot) {
		return nil, fmt.Errorf("path not exists or not folder: %s", contribRoot)
	}

	var err error
	contribRoot, err = filepath.Abs(contribRoot)
	if err != nil {
		return nil, err
	}

	var out []string
	err = filepath.WalkDir(contribRoot, func(osPathname string, de fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if de.IsDir() {
			for _, n := range blacklist.NoCodeDirs {
				if n == de.Name() {
					return filepath.SkipDir
				}
			}
			return nil
		}

		if de.Name() != "ya.make" {
			return nil
		}

		if !IsContribYMake(arcadiaRoot, osPathname) {
			return nil
		}

		out = append(out, filepath.Dir(osPathname))
		return nil
	})

	if err != nil {
		return nil, err
	}
	simplelog.Info("Targets collected", "n", len(out))
	simplelog.Warn("No versions", "libs", reflect.ValueOf(noVersions).MapKeys())

	return out, nil
}

func IsContribYMake(arcadiaRoot string, targetPath string) bool {
	yaMake, err := parseYMake(targetPath)
	switch {
	case err == errNotCppModule:
		// that's fine, just not a cpp module
		return false
	case os.IsNotExist(err):
		// that's fine
		return false
	case err != nil:
		simplelog.Error("failed to parse ya.make", "path", targetPath, "err", err)
		return false
	}
	if yaMake.Version == "" {
		libpath := filepath.Join(arcadiaRoot, arcadiaContribPrefix, "*", "ya.make")
		if matched, _ := filepath.Match(libpath, targetPath); matched {
			noVersions.Add(targetPath)
			simplelog.Debug("No version for " + targetPath)
		}
	} else {
		simplelog.Debug("Has version", "v", yaMake.Version, "target", targetPath)
	}

	return yaMake.Version != ""
}

func parseYMake(targetPath string) (*yamake.YMake, error) {
	yaMake, err := ya.ParseYaMake(targetPath)
	if err != nil {
		return nil, err
	}

	switch yaMake.TargetType {
	case yamake.TargetGenericProgram, yamake.TargetGenericLibrary:
		return yaMake, nil
	}

	return nil, errNotCppModule
}
