package golang

import (
	"fmt"
	"io/fs"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/yo/pkg/yamake"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/blacklist"
)

func FindTargets(arcadiaRoot string) ([]string, error) {
	simplelog.Info("parse vendor.policy")
	// read vendor policy and extract allowed packages
	vendorPolicyPath := filepath.Join(arcadiaRoot, "build", "rules", "go", "vendor.policy")
	directives, err := yamake.ReadPolicy(vendorPolicyPath, yamake.PolicyDirectiveAllow)
	if err != nil {
		return nil, fmt.Errorf("cannot read vendor.policy: %w", err)
	}

	vendorPath := filepath.Join(arcadiaRoot, "vendor")
	if _, err := os.Stat(vendorPath); err != nil {
		return nil, fmt.Errorf("failed to find arcadia vendor path: %w", err)
	}

	simplelog.Info("parse root modules")
	allPkgs := make(map[string]struct{})
	for _, directive := range directives {
		if directive.Type != yamake.PolicyDirectiveAllow {
			continue
		}

		if !strings.HasPrefix(directive.Package, vendorPrefix) {
			simplelog.Info("ignore non-vendor pkg", "pkg", directive.Package)
			continue
		}

		requiredPkg := strings.Trim(directive.Package[len(vendorPrefix):], "/")
		if _, ok := allPkgs[requiredPkg]; ok {
			continue
		}

		if findModuleBackward(vendorPath, requiredPkg) != "" {
			allPkgs[requiredPkg] = struct{}{}
			continue
		}

		simplelog.Debug("failed to find module in backward mode, try forward", "required_pkg", requiredPkg)
		possiblePkgs := findModuleForward(vendorPath, requiredPkg)
		if len(possiblePkgs) > 0 {
			for _, p := range possiblePkgs {
				simplelog.Debug("found pkg in forward mode", "required_pkg", requiredPkg, "pkg", p)
				allPkgs[p] = struct{}{}
			}
			continue
		}

		simplelog.Error("failed to find module", "required_pkg", requiredPkg)
	}

	pkgs := make([]string, 0, len(allPkgs))
	for requiredPkg := range allPkgs {
		pkgs = append(pkgs, requiredPkg)
	}

	return pkgs, nil
}

func findModuleBackward(root, pkgName string) string {
	isModuleDir := func(pkgPath string) bool {
		_, err := os.Stat(filepath.Join(pkgPath, fileSnapshotJSON))
		return err == nil
	}

	root = filepath.Clean(root)
	current := filepath.Clean(filepath.Join(root, pkgName))
	for {
		if isModuleDir(current) {
			return FilepathToPkgName(root, current)
		}

		next := filepath.Dir(current)
		if next == current || next == root {
			return ""
		}

		current = next
	}
}

func findModuleForward(root, pkgName string) []string {
	root = filepath.Clean(root)
	target := filepath.Clean(filepath.Join(root, pkgName))
	var out []string
	err := filepath.WalkDir(target, func(osPathname string, de fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if de.IsDir() {
			for _, n := range blacklist.NoCodeDirs {
				if n == de.Name() {
					return filepath.SkipDir
				}
			}
			return nil
		}

		if de.Name() == fileSnapshotJSON {
			out = append(out, FilepathToPkgName(root, filepath.Dir(osPathname)))
		}
		return nil
	})

	if err != nil {
		simplelog.Error("failed to find module in fwd mode", "pkg", pkgName, "err", err)
		return nil
	}
	return out
}

func FilepathToPkgName(root, filepath string) string {
	newPkg := strings.TrimPrefix(filepath, root)
	newPkg = strings.Trim(newPkg, "/\\")
	newPkg = strings.ReplaceAll(newPkg, "\\", "/")
	return path.Clean(newPkg)
}

func haveGoFiles(pkgPath string) bool {
	dir, err := ioutil.ReadDir(pkgPath)
	if err != nil {
		return false
	}

	for _, f := range dir {
		if filepath.Ext(f.Name()) == ".go" {
			return true
		}
	}

	return false
}
