package golang

import (
	"errors"
	"fmt"
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/slices"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
)

const (
	vendorPrefix = "vendor/"
	language     = "golang"
	name         = "arcadia-golang"
)

var _ manager.Manager = (*Manager)(nil)
var errSkipDep = errors.New("skip dep")
var contribOwners = []string{"g:go-contrib"}

type (
	ManagerOpts struct {
		// Arcadia path
		ArcadiaPath string

		// Parse test dependency
		WithTests bool

		// Don't process transitive deps
		WithoutTransitive bool
	}

	Manager struct {
		arcadiaPath    string
		vendorPath     string
		targets        []string
		targetPos      int
		curErr         error
		curModule      manager.Module
		withTests      bool
		withTransitive bool
		depsCache      map[string]manager.Module
	}
)

func NewManager(targets []string, opts ManagerOpts) (*Manager, error) {
	vendorPath := filepath.Join(opts.ArcadiaPath, "vendor")
	if _, err := os.Stat(vendorPath); err != nil {
		return nil, fmt.Errorf("failed to find arcadia vendor path: %w", err)
	}

	return &Manager{
		arcadiaPath:    opts.ArcadiaPath,
		vendorPath:     vendorPath,
		targets:        targets,
		targetPos:      -1,
		withTests:      opts.WithTests,
		withTransitive: !opts.WithoutTransitive,
		depsCache:      make(map[string]manager.Module),
	}, nil
}

func (m Manager) Name() string {
	return name
}

func (m *Manager) NextModule() bool {
	m.targetPos++
	if m.targetPos >= len(m.targets) {
		return false
	}

	requiredPkg := m.targets[m.targetPos]
	pkg, err := m.resolvePkg(requiredPkg)
	if err != nil {
		simplelog.Error("failed to resolve pkg", "pkg", requiredPkg, "err", err)
		return m.NextModule()
	}

	m.curModule = pkg
	return true
}

func (m *Manager) Err() error {
	return m.curErr
}

func (m *Manager) Module() manager.Module {
	return m.curModule
}

func (m *Manager) resolvePkg(pkgName string) (manager.Module, error) {
	module := manager.Module{
		Name:     pkgName,
		Language: language,
		// don't need to parse owners for Golang, it's always "g:go-contrib" (sorry bros)
		Owners: contribOwners,
	}

	simplelog.Debug("resolve pkg", "pkg", pkgName)
	moduleName := findModuleBackward(m.vendorPath, pkgName)
	if moduleName == "" {
		return module, fmt.Errorf("failed to find pkg: %s", pkgName)
	}

	module.LocalPath = filepath.Join(m.vendorPath, moduleName)
	snapshot, err := parseSnapshot(module.LocalPath)
	if err != nil {
		return module, fmt.Errorf("failed to parse pkg '%s': %w", moduleName, err)
	}

	simplelog.Debug("resolved pkg", "pkg", pkgName, "module_path", module.LocalPath, "version", snapshot.Version)
	module.Version, err = versionarium.NewVersion(language, snapshot.Version)
	if err != nil {
		return module, fmt.Errorf("failed to parse pkg '%s' version '%s': %w", pkgName, snapshot.Version, err)
	}

	if !m.withTransitive {
		return module, nil
	}

	allDeps := make(map[string]struct{})
	collectDeps := func(pkgs []string, isDev bool) error {
		for _, pkg := range pkgs {
			if pkg == pkgName {
				continue
			}

			if _, ok := allDeps[pkg]; ok {
				continue
			}
			allDeps[pkg] = struct{}{}

			subMod, err := m.resolveDependency(pkg, nil)
			if err != nil {
				if err == errSkipDep {
					// probably this is circular dep
					continue
				}
				return err
			}

			module.Dependencies = append(module.Dependencies, subMod)
		}
		return nil
	}

	for pkg, info := range snapshot.Packages {
		if pkg != pkgName && !strings.HasPrefix(pkg, pkgName+"/") {
			simplelog.Debug("ignore non-imported pkg", "pkg", pkg, "pkg_name", pkgName)
			continue
		}

		err = collectDeps(info.Imports, false)
		if err != nil {
			simplelog.Error("failed to resolve dependencies", "pkg", pkg, "err", err.Error())
		}

		if m.withTests {
			err = collectDeps(info.TestImports, true)
			if err != nil {
				simplelog.Error("failed to resolve test dependencies", "pkg", pkg, "err", err.Error())
			}
		}
	}

	return module, nil
}

func (m *Manager) resolveDependency(pkgName string, parents []string) (manager.Module, error) {
	if slices.ContainsString(parents, pkgName) {
		simplelog.Debug("circular recursion", "pkg", pkgName, "path", strings.Join(parents, " -> "))
		return manager.Module{}, errSkipDep
	}

	pkgPath := filepath.Join(m.vendorPath, pkgName)
	if !fsutils.IsDirExists(pkgPath) {
		// that's fine, we may have incomplete pkg in arcadia vendor
		simplelog.Debug("ignore non-exists pkg", "pkg", pkgName)
		return manager.Module{}, errSkipDep
	}

	if !haveGoFiles(pkgPath) {
		// that's fine, we may have incomplete pkg in arcadia vendor
		simplelog.Debug("ignore non-exists pkg", "pkg", pkgName)
		return manager.Module{}, errSkipDep
	}

	if dep, ok := m.depsCache[pkgName]; ok {
		return dep, nil
	}

	dep := manager.Module{
		Name:     pkgName,
		Language: language,
		// don't need to parse owners for Golang, it's always "g:go-contrib" (sorry bros)
		Owners: contribOwners,
	}

	simplelog.Debug("resolve dependency", "pkg", pkgName)
	moduleName := findModuleBackward(m.vendorPath, pkgName)
	if moduleName == "" {
		return dep, fmt.Errorf("failed to find dependency: %s", pkgName)
	}

	dep.LocalPath = filepath.Join(m.vendorPath, moduleName)
	snapshot, err := parseSnapshot(dep.LocalPath)
	if err != nil {
		return dep, fmt.Errorf("failed to parse module '%s': %w", moduleName, err)
	}

	simplelog.Debug("resolved dependency", "pkg", pkgName, "module_path", dep.LocalPath, "version", snapshot.Version)
	dep.Version, err = versionarium.NewVersion(language, snapshot.Version)
	if err != nil {
		return dep, fmt.Errorf("failed to parse pkg '%s' version '%s': %w", moduleName, snapshot.Version, err)
	}

	newParents := append(parents, pkgName)
	allDeps := make(map[string]struct{})
	collectDeps := func(pkgs []string, isDev bool) error {
		for _, pkg := range pkgs {
			if pkg == moduleName {
				continue
			}

			if _, ok := allDeps[pkg]; ok {
				continue
			}
			allDeps[pkg] = struct{}{}

			subMod, err := m.resolveDependency(pkg, newParents)
			if err != nil {
				if err == errSkipDep {
					// probably this is circular dep
					continue
				}
				return err
			}

			dep.Dependencies = append(dep.Dependencies, subMod)
		}
		return nil
	}

	if info, ok := snapshot.Packages[pkgName]; ok {
		err = collectDeps(info.Imports, false)
		if err != nil {
			simplelog.Error("failed to resolve dependencies", "pkg", pkgName, "err", err.Error())
		}

		if m.withTests {
			err = collectDeps(info.TestImports, true)
			if err != nil {
				simplelog.Error("failed to resolve test dependencies", "pkg", pkgName, "err", err.Error())
			}
		}
	}

	m.depsCache[pkgName] = dep
	return dep, nil
}
