package golang

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"path/filepath"
	"strings"
)

const (
	fileSnapshotJSON = ".yo.snapshot.json"
)

type PkgImports struct {
	Imports     []string
	TestImports []string
}

type SnapshotModule struct {
	Version  string
	Sum      string
	Packages map[string]PkgImports
}

var snapshotCache = make(map[string]SnapshotModule)

func parseSnapshot(pkgRoot string) (*SnapshotModule, error) {
	if mj, ok := snapshotCache[pkgRoot]; ok {
		return &mj, nil
	}

	cleanupImports := func(src []string) []string {
		var out []string
		for _, imp := range src {
			if !strings.ContainsRune(imp, '.') {
				// not external pkg
				continue
			}

			out = append(out, imp)
		}
		return out
	}

	// TODO(buglloc): .yo.snapshot.json is just a cache and may not be exists
	jsonPath := filepath.Join(pkgRoot, fileSnapshotJSON)
	jsonData, err := ioutil.ReadFile(jsonPath)
	if err != nil {
		return nil, err
	}

	var mj SnapshotModule
	if err := json.Unmarshal(jsonData, &mj); err != nil {
		return nil, fmt.Errorf("error in file '%s': %w", jsonPath, err)
	}

	pkgs := make(map[string]PkgImports, len(mj.Packages))
	for importPath, pkg := range mj.Packages {
		pkgs[importPath] = PkgImports{
			Imports:     cleanupImports(pkg.Imports),
			TestImports: cleanupImports(pkg.TestImports),
		}
	}

	mj.Packages = pkgs
	snapshotCache[pkgRoot] = mj
	return &mj, nil
}
