package java

import (
	"errors"
	"fmt"
	"io/fs"
	"io/ioutil"
	"path/filepath"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yamake"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/blacklist"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
)

var errNotJavaModule = errors.New("not java module")

func FindTargets(arcadiaRoot string) ([]string, error) {
	contribRoot := filepath.Join(arcadiaRoot, "contrib", "java")
	if !fsutils.IsDirExists(contribRoot) {
		return nil, fmt.Errorf("path not exists or not folder: %s", contribRoot)
	}

	var err error
	contribRoot, err = filepath.Abs(contribRoot)
	if err != nil {
		return nil, err
	}

	var out []string
	err = filepath.WalkDir(contribRoot, func(osPathname string, de fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if de.IsDir() {
			for _, n := range blacklist.NoCodeDirs {
				if n == de.Name() {
					return filepath.SkipDir
				}
			}
			return nil
		}

		if de.Name() != "ya.make" {
			return nil
		}

		if !IsContribYMake(osPathname) {
			return nil
		}

		contribDir := filepath.Dir(osPathname)
		files, err := ioutil.ReadDir(contribDir)
		if err != nil {
			simplelog.Error("failed to list contrib directory", "path", contribDir, "err", err)
			return nil
		}

		for _, f := range files {
			if !f.IsDir() {
				continue
			}

			out = append(out, filepath.Join(contribDir, f.Name()))
		}
		return nil
	})

	if err != nil {
		return nil, err
	}
	return out, nil
}

func IsContribYMake(targetPath string) bool {
	_, err := parseYMake(targetPath)
	switch {
	case err == errNotJavaModule:
		// that's fine, just not a java module
		return false
	case err != nil:
		simplelog.Error("failed to parse ya.make", "path", targetPath, "err", err)
		return false
	}

	return true
}

//TODO(melkikh): think about
// https://a.yandex-team.ru/arc_vcs/contrib/java/ru/yandex/market/logistics/management/logistics-management-service-client/1.0.96/ya.make
//IF(YMAKE_JAVA_MODULES)
//    JAVA_CONTRIB()
//ELSE()
//    JAVA_LIBRARY()
//ENDIF()

func parseYMake(targetPath string) (*yamake.YMake, error) {
	yaMake, err := ya.ParseYaMake(targetPath)
	if err != nil {
		return nil, err
	}

	switch yaMake.TargetType {
	case yamake.TargetJavaLibrary, yamake.TargetGenericLibrary:
		return yaMake, nil
	}

	return nil, errNotJavaModule
}
