package java

import (
	"fmt"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
)

const (
	name     = "arcadia-java"
	language = "java"
)

type (
	ManagerOpts struct {
		ArcadiaPath string
	}

	Manager struct {
		arcadiaRoot string
		targets     []string
		targetPos   int
		curModule   manager.Module
		curErr      error
	}

	javaModule struct {
		Name      string
		Version   string
		LocalPath string
	}
)

var _ manager.Manager = (*Manager)(nil)

func NewManager(targets []string, opts ManagerOpts) (*Manager, error) {
	return &Manager{
		arcadiaRoot: opts.ArcadiaPath,
		targets:     targets,
		targetPos:   -1,
	}, nil
}

func (m Manager) Name() string {
	return name
}

func (m *Manager) NextModule() bool {
	m.targetPos++
	if m.targetPos >= len(m.targets) {
		return false
	}

	target := m.targets[m.targetPos]
	if !filepath.IsAbs(target) {
		target = filepath.Join(m.arcadiaRoot, target)
	}

	simplelog.Debug("parse target", "target", target)
	jModule := parseModulePath(target)
	if jModule.Name == "" {
		simplelog.Error("failed to parse target", "target", target, "err", "can't parse module from its path")
		return m.NextModule()
	}

	version, err := versionarium.NewVersion(language, jModule.Version)
	if err != nil {
		simplelog.Error("failed to parse target version", "target", target, "version", jModule.Version, "err", err)
		return m.NextModule()
	}

	m.curModule = manager.Module{
		Name:      jModule.Name,
		Version:   version,
		LocalPath: jModule.LocalPath,
		Language:  language,
		Owners:    ya.ModuleOwners(jModule.LocalPath),
	}
	return true
}

func (m *Manager) Err() error {
	return m.curErr
}

func (m *Manager) Module() manager.Module {
	return m.curModule
}

func parseModulePath(modulePath string) javaModule {
	normPath := strings.ReplaceAll(modulePath, "\\", "/")
	idx := strings.LastIndex(normPath, "contrib/java/")
	if idx != -1 {
		normPath = normPath[idx+len("contrib/java/"):]
	}

	parts := strings.Split(normPath, "/")
	if len(parts) < 3 {
		return javaModule{}
	}

	version := parts[len(parts)-1]
	artifactID := parts[len(parts)-2]
	groupID := strings.Join(parts[:len(parts)-2], ".")

	return javaModule{
		Name:      fmt.Sprintf("%s:%s", groupID, artifactID),
		Version:   version,
		LocalPath: modulePath,
	}
}
