package python

import (
	"errors"
	"fmt"
	"io/fs"
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yamake"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/blacklist"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
)

var errNotPythonModule = errors.New("not python module")

func FindTargets(arcadiaRoot string) ([]string, error) {
	contribRoot := filepath.Join(arcadiaRoot, "contrib", "python")
	if !fsutils.IsDirExists(contribRoot) {
		return nil, fmt.Errorf("path not exists or not folder: %s", contribRoot)
	}

	var err error
	contribRoot, err = filepath.Abs(contribRoot)
	if err != nil {
		return nil, err
	}

	var out []string
	err = filepath.WalkDir(contribRoot, func(osPathname string, de fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if de.IsDir() {
			for _, n := range blacklist.NoCodeDirs {
				if n == de.Name() {
					return filepath.SkipDir
				}
			}
			return nil
		}

		if de.Name() != "ya.make" {
			return nil
		}

		if !IsContribYMake(osPathname) {
			return nil
		}

		out = append(out, filepath.Dir(osPathname))
		return nil
	})

	if err != nil {
		return nil, err
	}
	return out, nil
}

func IsContribYMake(targetPath string) bool {
	yaMake, err := parseYMake(targetPath)
	switch {
	case err == errNotPythonModule:
		// that's fine, just not a python module
		return false
	case os.IsNotExist(err):
		// that's fine
		return false
	case err != nil:
		simplelog.Error("failed to parse ya.make", "path", targetPath, "err", err)
		return false
	}

	return yaMake.Version != ""
}

func parseYMake(targetPath string) (*yamake.YMake, error) {
	yaMake, err := ya.ParseYaMake(targetPath)
	if err != nil {
		return nil, err
	}

	switch yaMake.TargetType {
	case yamake.TargetGenericProgram, yamake.TargetGenericLibrary, yamake.TargetPyProgram, yamake.TargetPyLibrary:
		return yaMake, nil
	}

	return nil, errNotPythonModule
}

func guessModuleName(modulePath string) string {
	c := strings.LastIndex(modulePath, "contrib/python") + 15
	i := strings.Index(modulePath[c:], "/")
	if i <= 0 {
		i = len(modulePath)
	} else {
		i += c
	}

	return modulePath[c:i]
}
