package python

import (
	"fmt"
	"path/filepath"
	"regexp"

	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	yadiManager "a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip"
)

const (
	name     = "arcadia-python"
	language = "python"
)

type (
	ManagerOpts struct {
		ArcadiaPath string
	}

	Manager struct {
		pip       *pip.Manager
		targets   []string
		targetPos int
		curModule manager.Module
		curErr    error
	}
)

var versionParser = regexp.MustCompile(`^v?((?:\d+)(?:\.\d+)?(?:\.\d+)?)`)
var _ manager.Manager = (*Manager)(nil)

func NewManager(targets []string, _ ManagerOpts) (*Manager, error) {
	parent, err := pip.NewManager(pip.ManagerOpts{
		WithDev:          false,
		ResolveMode:      yadiManager.ResolveRemote,
		WithoutPypiCache: true,
	})
	if err != nil {
		return nil, err
	}

	return &Manager{
		pip:       parent,
		targets:   targets,
		targetPos: -1,
	}, nil
}

func (m Manager) Name() string {
	return name
}

func (m *Manager) NextModule() bool {
	m.targetPos++
	if m.targetPos >= len(m.targets) {
		return false
	}

	parseVersion := func(version string) (versionarium.Version, error) {
		parts := versionParser.FindStringSubmatch(version)
		if len(parts) == 0 {
			return nil, fmt.Errorf("version ('%s') don't match: %s", version, versionParser.String())
		}

		parsed, err := versionarium.NewVersion(language, parts[1])
		if err != nil {
			return nil, fmt.Errorf("failed to parse module version (%s): %w", version, err)
		}
		return parsed, nil
	}

	target := m.targets[m.targetPos]
	modulePath := filepath.Join(target, "ya.make")
	yaMake, err := parseYMake(modulePath)
	if err != nil {
		simplelog.Warn("failed to parse ya.make", "path", modulePath, "err", err)
		return m.NextModule()
	}

	version, err := parseVersion(yaMake.Version)
	if err != nil {
		simplelog.Warn("failed to parse module version", "path", modulePath, "err", err)
		return m.NextModule()
	}

	name := yaMake.Name
	if name == "" {
		name = guessModuleName(modulePath)
	}

	m.curModule = manager.Module{
		Name:      pypipkg.NormalizeName(name),
		LocalPath: filepath.Dir(modulePath),
		Version:   version,
		Language:  language,
		Owners:    yaMake.Owners,
	}
	return true
}

func (m *Manager) Err() error {
	return m.curErr
}

func (m *Manager) Module() manager.Module {
	return m.curModule
}
