package yamake

import (
	"io/fs"
	"path"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yamake"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/blacklist"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
)

func FindTargets(root string, target string) ([]YMakeTarget, error) {
	var out []YMakeTarget
	err := filepath.WalkDir(target, func(osPathname string, de fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if de.IsDir() {
			for _, n := range blacklist.NoCodeDirs {
				if n == de.Name() {
					return filepath.SkipDir
				}
			}
			return nil
		}

		if de.Name() != "ya.make" {
			return nil
		}

		yaMake, err := ya.ParseYaMake(osPathname)
		if err != nil {
			simplelog.Error("failed to parse ya.make", "path", osPathname, "err", err)
			return nil
		}

		var targetType TargetType
		switch yaMake.TargetType {
		case yamake.TargetGenericProgram, yamake.TargetGenericUDF, yamake.TargetGenericDLL:
			targetType = TargetTypeCpp
		case yamake.TargetGoProgram, yamake.TargetGoDLL:
			targetType = TargetTypeGolang
		case yamake.TargetPyProgram, yamake.TargetPythonUDF:
			targetType = TargetTypePython
		case yamake.TargetJavaProgram:
			targetType = TargetTypeJava
		default:
			return nil
		}

		targetPath, err := filepath.Abs(filepath.Dir(osPathname))
		if err != nil {
			simplelog.Error("failed to normalize module path", "path", osPathname, "err", err)
			return nil
		}

		out = append(out, YMakeTarget{
			Path:      targetPath,
			ModuleDir: FilepathToModule(root, targetPath),
			Type:      targetType,
		})
		return nil
	})

	return out, err
}

func FilepathToModule(root, filepath string) string {
	newPkg := strings.TrimPrefix(filepath, root)
	newPkg = strings.Trim(newPkg, "/\\")
	newPkg = strings.ReplaceAll(newPkg, "\\", "/")
	return path.Clean(newPkg)
}
