package yamake

import (
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/cpp"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/python"
)

type langOption struct {
	Language       string
	RootVersion    versionarium.Version
	IsContrib      func(modulePath string) bool
	FixContribPath func(root, modulePath string) string
}

var langOptions = map[TargetType]langOption{
	TargetTypePython: {
		Language:    "python",
		IsContrib:   newPathPrefixMatch("contrib/python/"),
		RootVersion: versionarium.MustNewVersion("python", "0.0.0"),
		FixContribPath: func(root, modulePath string) string {
			isModuleDir := func(pkgPath string) bool {
				if st, err := os.Stat(pkgPath); err != nil || !st.IsDir() {
					return false
				}

				return python.IsContribYMake(filepath.Join(pkgPath, "ya.make"))
			}

			current := filepath.Clean(filepath.Join(root, modulePath))
			for {
				if isModuleDir(current) {
					return current
				}

				next := filepath.Dir(current)
				if next == current || next == root {
					return ""
				}

				current = next
			}
		},
	},
	TargetTypeGolang: {
		Language:    "golang",
		IsContrib:   newPathPrefixMatch("vendor/"),
		RootVersion: versionarium.MustNewVersion("golang", "v0.0.0"),
		FixContribPath: func(root, modulePath string) string {
			return strings.TrimPrefix(modulePath, "vendor/")
		},
	},
	TargetTypeJava: {
		Language: "java",
		IsContrib: func(modulePath string) bool {
			return strings.HasPrefix(modulePath, "contrib/java/") && strings.HasSuffix(modulePath, ".jar")
		},
		RootVersion: versionarium.MustNewVersion("java", "0.0.0"),
		FixContribPath: func(root, modulePath string) string {
			return modulePath
		},
	},
	TargetTypeCpp: {
		Language:    "cpp",
		IsContrib:   newPathPrefixMatch("contrib/libs/"),
		RootVersion: versionarium.MustNewVersion("cpp", "0.0.0"),
		FixContribPath: func(root, modulePath string) string {
			isModuleDir := func(pkgPath string) bool {
				if st, err := os.Stat(pkgPath); err != nil || !st.IsDir() {
					return false
				}

				return cpp.IsContribYMake(root, filepath.Join(pkgPath, "ya.make"))
			}

			current := filepath.Clean(filepath.Join(root, modulePath))
			for {
				if isModuleDir(current) {
					return current
				}

				next := filepath.Dir(current)
				if next == current || next == root {
					return ""
				}

				current = next
			}
		},
	},
}

func newPathPrefixMatch(prefix string) func(modulePath string) bool {
	return func(modulePath string) bool {
		return strings.HasPrefix(modulePath, prefix)
	}
}
