package yamake

import (
	"bufio"
	"bytes"
	"encoding/json"
	"fmt"
	"path"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-arc/internal/ya"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/cpp"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/golang"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/java"
	"a.yandex-team.ru/security/yadi/yadi-arc/pkg/manager/python"
)

const (
	name = "yamake"
)

var _ manager.Manager = (*Manager)(nil)

type (
	ManagerOpts struct {
		ArcadiaPath string
		BatchSize   int
	}

	Manager struct {
		arcadiaPath       string
		targets           []YMakeTarget
		targetPos         int
		resolvedTargets   []ResolvedTarget
		resolvedTargetPos int
		maxTargets        int
		curModule         manager.Module
		curErr            error
	}

	yaGraphJSON struct {
		Result []string `json:"result"`
		Graph  []struct {
			Inputs []string `json:"inputs"`
			Type   int      `json:"type"`
			UID    string   `json:"uid"`
			Target struct {
				ModuleDir string `json:"module_dir"`
			} `json:"target_properties"`
		} `json:"graph"`
	}
)

func NewManager(targets []YMakeTarget, opts ManagerOpts) (*Manager, error) {
	return &Manager{
		targets:           targets,
		targetPos:         -1,
		resolvedTargetPos: -1,
		maxTargets:        opts.BatchSize,
		arcadiaPath:       opts.ArcadiaPath,
	}, nil
}

func (m Manager) Name() string {
	return name
}

func (m *Manager) NextModule() bool {
	newManager := func(targetType TargetType, contribs []string) (manager.Manager, error) {
		switch targetType {
		case TargetTypePython:
			return python.NewManager(contribs, python.ManagerOpts{
				ArcadiaPath: m.arcadiaPath,
			})
		case TargetTypeGolang:
			return golang.NewManager(contribs, golang.ManagerOpts{
				ArcadiaPath: m.arcadiaPath,
				WithTests:   false,
				// We don't need to check transitive deps, due to ya make already returns it in the flat list
				WithoutTransitive: true,
			})
		case TargetTypeJava:
			return java.NewManager(contribs, java.ManagerOpts{
				ArcadiaPath: m.arcadiaPath,
			})
		case TargetTypeCpp:
			return cpp.NewManager(contribs, cpp.ManagerOpts{
				ArcadiaPath: m.arcadiaPath,
			})
		default:
			return nil, fmt.Errorf("unsupported program type: %d", targetType)
		}
	}

	target, ok := m.nextResolvedTarget()
	if !ok {
		return false
	}

	pm, err := newManager(target.Type, target.Contribs)
	if err != nil {
		simplelog.Error("failed to create new manager", "target", target.Path, "err", err)
		return m.NextModule()
	}

	module := manager.Module{
		Name:      strings.TrimPrefix(target.Path, m.arcadiaPath),
		Version:   langOptions[target.Type].RootVersion,
		LocalPath: target.Path,
		Owners:    ya.ModuleOwners(target.Path),
		Language:  langOptions[target.Type].Language,
	}

	for pm.NextModule() {
		module.Dependencies = append(module.Dependencies, pm.Module())
	}

	if err = pm.Err(); err != nil {
		simplelog.Error("failed to iterate modules", "target", target.Path, "err", err)
	}

	m.curModule = module
	return true
}

func (m *Manager) Err() error {
	return m.curErr
}

func (m *Manager) Module() manager.Module {
	return m.curModule
}

func (m *Manager) nextResolvedTarget() (ResolvedTarget, bool) {
	if m.curErr != nil {
		return ResolvedTarget{}, false
	}

	m.resolvedTargetPos++
	if m.resolvedTargetPos < len(m.resolvedTargets) {
		return m.resolvedTargets[m.resolvedTargetPos], true
	}

	toResolve := make(map[TargetType][]YMakeTarget)
	counts := 0
	for {
		t, ok := m.nextTarget()
		if !ok {
			break
		}

		toResolve[t.Type] = append(toResolve[t.Type], t)
		counts++
		if counts >= m.maxTargets {
			break
		}
	}

	if len(toResolve) == 0 {
		return ResolvedTarget{}, false
	}

	m.resolvedTargetPos = -1
	for typ, targets := range toResolve {
		resolved, err := m.resolveTargets(typ, targets...)
		if err != nil {
			m.curErr = fmt.Errorf("unable to resolve targets: %w", err)
			break
		}

		m.resolvedTargets = append(m.resolvedTargets, resolved...)
	}

	return m.nextResolvedTarget()
}

func (m *Manager) nextTarget() (YMakeTarget, bool) {
	m.targetPos++
	if m.targetPos >= len(m.targets) {
		return YMakeTarget{}, false
	}

	return m.targets[m.targetPos], true
}

func (m *Manager) resolveTargets(typ TargetType, targets ...YMakeTarget) ([]ResolvedTarget, error) {
	switch typ {
	case TargetTypeJava:
		return resolveJavaTargets(m.arcadiaPath, targets...)
	default:
		return resolveBuildGraphTargets(m.arcadiaPath, targets...)
	}
}

func resolveBuildGraphTargets(arcadiaPath string, targets ...YMakeTarget) ([]ResolvedTarget, error) {
	args := []string{
		"make",
		"--threads=0",
		"--dump-json-graph",
		"--keep-going",
		"--build=release",
		"--target",
	}

	for _, t := range targets {
		args = append(args, t.ModuleDir)
	}

	data, err := ya.Run(args, nil, arcadiaPath)
	if err != nil {
		return nil, err
	}

	mineInputs := func(opts langOption, inputs []string) []string {
		var (
			collected = make(map[string]struct{})
			out       []string
		)

		for _, i := range inputs {
			var input string
			switch {
			case strings.HasPrefix(i, "$(SOURCE_ROOT)/"):
				input = i[len("$(SOURCE_ROOT)/"):]
			case strings.HasPrefix(i, "$(BUILD_ROOT)/"):
				input = i[len("$(BUILD_ROOT)/"):]
			default:
				continue
			}

			if !opts.IsContrib(input) {
				continue
			}

			if _, ok := collected[input]; ok {
				continue
			}

			collected[input] = struct{}{}

			input = path.Dir(input)
			contrib := opts.FixContribPath(arcadiaPath, input)
			if contrib == "" {
				simplelog.Warn("can't determine contrib module from path", "input", input)
				continue
			}

			if _, ok := collected[contrib]; ok {
				continue
			}

			collected[contrib] = struct{}{}
			out = append(out, contrib)
		}
		return out
	}

	var yaGraph yaGraphJSON
	err = json.Unmarshal(data.Stdout(), &yaGraph)
	if err != nil {
		return nil, fmt.Errorf("failed to parse ya graph: %w", err)
	}

	out := make([]ResolvedTarget, 0, len(targets))
	for _, r := range yaGraph.Graph {
		for _, t := range targets {
			if r.Target.ModuleDir != t.ModuleDir {
				continue
			}

			out = append(out, ResolvedTarget{
				YMakeTarget: t,
				Contribs:    mineInputs(langOptions[t.Type], r.Inputs),
			})
			break
		}
	}
	return out, nil
}

func resolveJavaTargets(arcadiaPath string, targets ...YMakeTarget) ([]ResolvedTarget, error) {
	args := []string{
		"java",
		"classpath",
		"-r",
		"--target",
	}

	for _, t := range targets {
		args = append(args, t.ModuleDir)
	}

	data, err := ya.Run(args, nil, arcadiaPath)
	if err != nil {
		return nil, err
	}

	mineInputs := func(opts langOption, inputs []string) []string {
		var (
			collected = make(map[string]struct{})
			out       []string
		)

		for _, input := range inputs {
			if !opts.IsContrib(input) {
				continue
			}

			if _, ok := collected[input]; ok {
				continue
			}

			collected[input] = struct{}{}

			input = path.Dir(input)
			contrib := opts.FixContribPath(arcadiaPath, input)
			if contrib == "" {
				simplelog.Warn("can't determine contrib module from path", "input", input)
				continue
			}

			if _, ok := collected[contrib]; ok {
				continue
			}

			collected[contrib] = struct{}{}
			out = append(out, contrib)
		}
		return out
	}

	var (
		inputs []string
		target string
	)
	out := make([]ResolvedTarget, 0, len(targets))
	processTarget := func() {
		for _, t := range targets {
			if target != t.ModuleDir {
				continue
			}

			out = append(out, ResolvedTarget{
				YMakeTarget: t,
				Contribs:    mineInputs(langOptions[t.Type], inputs),
			})
			break
		}
	}

	scanner := bufio.NewScanner(bytes.NewBuffer(data.Stdout()))
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if line == "" {
			continue
		}

		hasMark := strings.HasPrefix(line, "=== ")
		if hasMark && target != "" {
			processTarget()
			target = ""
		}

		switch {
		case hasMark && strings.HasSuffix(line, "@JAVA_PROGRAM[JAR_RUNABLE] ==="):
			target = line[4 : len(line)-30]
		case target != "":
			inputs = append(inputs, line)
		}
	}

	if target != "" {
		processTarget()
	}

	return out, scanner.Err()
}
