package cli

import (
	"fmt"
	"os"

	"golang.org/x/crypto/ssh/terminal"

	"a.yandex-team.ru/security/yadi/yadi-os/pkg/formatter"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/formatter/consolefmt"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/formatter/jsonfmt"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/formatter/protofmt"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/formatter/textfmt"
)

const (
	TextFormatterName    = "text"
	ConsoleFormatterName = "console"
	JSONFormatterName    = "json"
	ProtoFormatterName   = "proto"
)

func ListFormatter(format string) (formatterFn formatter.ListFn, err error) {
	switch format {
	case TextFormatterName:
		formatterFn = textfmt.ListFormatter
	case ConsoleFormatterName:
		formatterFn = consolefmt.ListFormatter
	case JSONFormatterName:
		formatterFn = jsonfmt.ListFormatter
	case ProtoFormatterName:
		formatterFn = protofmt.ListFormatter
	default:
		err = fmt.Errorf(
			"unrecognized format %q, must be one of: text, console, json or proto",
			format,
		)
	}
	return
}

func CheckPackagesFormatter(format string) (formatterFn formatter.CheckPackagesFn, err error) {
	switch format {
	case TextFormatterName:
		formatterFn = textfmt.CheckPackagesFormatter
	case ConsoleFormatterName:
		formatterFn = consolefmt.CheckPackagesFormatter
	case JSONFormatterName:
		formatterFn = jsonfmt.CheckPackagesFormatter
	case ProtoFormatterName:
		formatterFn = protofmt.CheckPackagesFormatter
	default:
		err = fmt.Errorf(
			"unrecognized format %q, must be one of: text, console, json or proto",
			format,
		)
	}
	return
}

func CheckKernelFormatter(format string) (formatterFn formatter.CheckKernelFn, err error) {
	switch format {
	case TextFormatterName:
		formatterFn = textfmt.CheckKernelFormatter
	case ConsoleFormatterName:
		formatterFn = consolefmt.CheckKernelFormatter
	case JSONFormatterName:
		formatterFn = jsonfmt.CheckKernelFormatter
	case ProtoFormatterName:
		formatterFn = protofmt.CheckKernelFormatter
	default:
		err = fmt.Errorf(
			"unrecognized format %q, must be one of: text, console, json or proto",
			format,
		)
	}
	return
}

func DetectDefaultFormatter() string {
	switch {
	case terminal.IsTerminal(int(os.Stdout.Fd())):
		return ConsoleFormatterName
	default:
		return TextFormatterName
	}
}
