package commands

import (
	"encoding/json"
	"fmt"
	"os"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hostctl/pkg/render/rtcutil"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/config"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/hostctl"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/mockmngr"
)

var hostCtlCmd = &cobra.Command{
	Use:   "hostctl",
	Short: "test RTC packages for any known vulnerabilities",
	RunE:  runTestRtcSalt,
}

var hostCtlArgs struct {
	Cluster     string
	UnitPath    string
	FixableOnly bool
}

func init() {
	flags := hostCtlCmd.PersistentFlags()
	flags.StringVar(&hostCtlArgs.Cluster, "cluster", "sas", "RTC cluster to use")
	flags.StringVar(&hostCtlArgs.UnitPath, "unit", "./units.d/upstream-packages.yaml", "path to packages spec")
	flags.BoolVar(&hostCtlArgs.FixableOnly, "fixable", false, "report only fixable issues")

	rootCmd.AddCommand(hostCtlCmd)
}

func runTestRtcSalt(_ *cobra.Command, _ []string) error {
	var targetCluster string
	for _, c := range rtcutil.GetClusterList() {
		if strings.EqualFold(c, hostCtlArgs.Cluster) {
			targetCluster = c
			break
		}
	}

	if targetCluster == "" {
		return fmt.Errorf("unknown RTC cluster: %s", hostCtlArgs.Cluster)
	}

	analyzer, err := analyze.New(analyze.Options{
		FeedURI:                config.FeedURI,
		MinimumSeverity:        config.MinimumSeverity,
		VulnerabilitiesExclude: rootOpts.SkipIssues,
		PackagesExclude:        rootOpts.SkipPackages,
		FixableOnly:            testPackagesArgs.FixableOnly,
	})

	if err != nil {
		return err
	}

	clusterPkgs, err := hostctl.Render(targetCluster, hostCtlArgs.UnitPath)
	if err != nil {
		return fmt.Errorf("unable to render pkgs on cluster %q: %w", targetCluster, err)
	}

	out := make([]hostctl.Result, 0)
	for _, clusterPkg := range clusterPkgs {
		checkResults, err := analyzer.CheckPackages(mockmngr.NewManager(mockmngr.ManagerInfo{
			Name:         "hostctl",
			Distributive: "ubuntu",
			//TODO(buglloc): retrieve me from hostctl
			DistributiveCodename: "xenial",
			Packages:             clusterPkg.Pkgs,
		}))
		if err != nil {
			return err
		}

		if len(checkResults) > 0 {
			out = append(out, hostctl.Result{
				Vulnerabilities: checkResults,
				Hosts:           clusterPkg.Hosts,
			})
		}
	}

	switch rootOpts.Format {
	case "json":
		err = json.NewEncoder(os.Stdout).Encode(out)
	case "text", "console":
		_, err = os.Stdout.Write(hostctl.FormatResults(out))
	default:
		return fmt.Errorf("unsupported output format: %s", rootOpts.Format)
	}

	if err != nil {
		return err
	}

	if len(out) > 0 {
		os.Exit(rootOpts.ExitCode)
	}
	return nil
}
