package commands

import (
	"encoding/json"
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/lineage"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/kernel"
)

var infoCmd = &cobra.Command{
	Use:   "info",
	Short: "gather short os info",
	RunE: func(_ *cobra.Command, args []string) error {
		kernelInfo, err := kernel.Current()
		if err != nil {
			return xerrors.Errorf("failed to get current kernel: %w", err)
		}

		currentOS, err := lineage.CurrentOS()
		if err != nil {
			return xerrors.Errorf("failed to detect OS: %w", err)
		}

		switch rootOpts.Format {
		case "json":
			type Kernel struct {
				Release string `json:"release"`
			}

			type OS struct {
				Family   string `json:"family"`
				Release  string `json:"release"`
				Codename string `json:"codename"`
			}

			out := struct {
				Kernel Kernel `json:"kernel"`
				OS     OS     `json:"os"`
			}{
				Kernel: Kernel{
					Release: kernelInfo.Release.String(),
				},
				OS: OS{
					Family:   currentOS.Family,
					Release:  currentOS.Release,
					Codename: currentOS.Codename,
				},
			}
			if err := json.NewEncoder(os.Stdout).Encode(out); err != nil {
				return xerrors.Errorf("failed to encode json output: %w", err)
			}
		default:
			fmt.Printf("kernel release: %s\n", kernelInfo.Release.String())
			fmt.Printf("os family: %s\n", currentOS.Family)
			fmt.Printf("os codename: %s\n", currentOS.Codename)
			fmt.Printf("os release: %s\n", currentOS.Release)
		}
		return nil
	},
}

func init() {
	rootCmd.AddCommand(infoCmd)
}
