package commands

import (
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/yadi/yadi-os/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/config"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/kernel"
)

var testKernelCmd = &cobra.Command{
	Use:   "kernel",
	Short: "test kernel for any known vulnerabilities",
	RunE:  runTestKernel,
}

var testKernelArgs struct {
	KernelVersion string
	FixableOnly   bool
}

func init() {
	flags := testKernelCmd.PersistentFlags()
	flags.StringVarP(&testKernelArgs.KernelVersion, "release", "r", "", "kernel release to test (format of \"uname -r\")")
	flags.BoolVar(&testKernelArgs.FixableOnly, "fixable", testKernelArgs.FixableOnly, "report only fixable issues")
	testCmd.AddCommand(testKernelCmd)
}

func runTestKernel(_ *cobra.Command, _ []string) error {
	var linux *kernel.Kernel
	analyzer, err := analyze.New(analyze.Options{
		FeedURI:                config.FeedURI,
		MinimumSeverity:        config.MinimumSeverity,
		VulnerabilitiesExclude: rootOpts.SkipIssues,
		PackagesExclude:        rootOpts.SkipPackages,
		FixableOnly:            testKernelArgs.FixableOnly,
	})

	if err != nil {
		return err
	}

	if testKernelArgs.KernelVersion == "" {
		linux, err = kernel.Current()
	} else {
		linux, err = kernel.Parse(testKernelArgs.KernelVersion)
	}

	if err != nil {
		return err
	}

	checkResults, err := analyzer.CheckKernel(*linux)
	if err != nil {
		return err
	}

	format, err := cli.CheckKernelFormatter(rootOpts.Format)
	if err != nil {
		return err
	}

	_, _ = os.Stdout.Write(format(checkResults))

	if len(checkResults) > 0 {
		os.Exit(rootOpts.ExitCode)
	}
	return nil
}
