package commands

import (
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/yadi-os/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/config"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager"
)

var testPackagesCmd = &cobra.Command{
	Use:   "packages",
	Short: "test OS packages for any known vulnerabilities",
	RunE:  runTestPackages,
}

var testPackagesArgs struct {
	FixableOnly bool
}

func init() {
	flags := testPackagesCmd.PersistentFlags()
	flags.BoolVar(&testPackagesArgs.FixableOnly, "fixable", testPackagesArgs.FixableOnly, "report only fixable issues")

	testCmd.AddCommand(testPackagesCmd)
}

func runTestPackages(_ *cobra.Command, args []string) error {
	analyzer, err := analyze.New(analyze.Options{
		FeedURI:                config.FeedURI,
		MinimumSeverity:        config.MinimumSeverity,
		VulnerabilitiesExclude: rootOpts.SkipIssues,
		PackagesExclude:        rootOpts.SkipPackages,
		FixableOnly:            testPackagesArgs.FixableOnly,
	})

	if err != nil {
		return err
	}

	pkgManager, err := pkgmanager.DetectPkgManager(rootOpts.Root)
	if err != nil {
		if rootOpts.SkipUnknownOS && xerrors.Is(err, pkgmanager.ErrUnsupportedOs) {
			// that's fine
			return nil
		}
		return err
	}

	checkResults, err := analyzer.CheckPackages(pkgManager)
	if err != nil {
		return err
	}

	format, err := cli.CheckPackagesFormatter(rootOpts.Format)
	if err != nil {
		return err
	}

	_, _ = os.Stdout.Write(format(checkResults))

	if len(checkResults) > 0 {
		os.Exit(rootOpts.ExitCode)
	}
	return nil
}
