package commands

import (
	"errors"
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi-os/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/config"
)

var (
	rootOpts struct {
		SeverityLevel   int
		Format          string
		FeedURI         string
		RawSkipIssues   string
		RawSkipPackages string
		Root            string
		SkipIssues      []string
		SkipPackages    []string
		SkipUnknownOS   bool
		Verbose         bool
		ShowVersion     bool
		ExitCode        int
	}
)

// rootCmd represents the base command when called without any subcommands
var rootCmd = &cobra.Command{
	Use:          "yadi-os",
	SilenceUsage: true,
	Short:        "Yandex Dependency Inspector for GNU/Linux",
	Long: `Yandex Dependency Inspector for GNU/Linux

Home page: https://yadi.yandex-team.ru/
Documentation: https://wiki.yandex-team.ru/product-security/yadi-os/
Vulnerability Database: https://yadi.yandex-team.ru/vulns`,
	PersistentPreRunE: rootPreRun,
}

// Execute adds all child commands to the root command sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	rootCmd.Version = config.FullVersion()
	rootCmd.SetVersionTemplate("{{.Version}}\n")

	if err := rootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func init() {
	flags := rootCmd.PersistentFlags()
	flags.StringVarP(&rootOpts.Format, "format", "f", cli.DetectDefaultFormatter(),
		"output format (text, console, json or proto)")
	flags.CountVarP(&rootOpts.SeverityLevel, "level", "l",
		"report issues of a given severity level or higher (-l for Low, -ll for Medium, -lll for High, -llll Critical) (default: Medium)")
	flags.StringSliceVar(&rootOpts.SkipIssues, "ignore-issue", []string{},
		"comma separated list of skipped issues")
	flags.StringSliceVar(&rootOpts.SkipPackages, "ignore-package", []string{},
		"comma separated list of skipped packages")
	flags.StringVar(&rootOpts.Root, "root", "/",
		"root file system path")
	flags.StringVar(&rootOpts.FeedURI, "feed", os.Getenv("YADIOS_FEED"),
		"vulnerability DB URI")
	flags.BoolVar(&rootOpts.SkipUnknownOS, "skip-unknown-os", false,
		"skip unknown OS")
	flags.BoolVar(&rootOpts.Verbose, "verbose", false,
		"verbose output")
	flags.BoolVar(&rootOpts.ShowVersion, "version", false,
		"show the current version")
	flags.IntVar(&rootOpts.ExitCode, "exit-code", 3,
		"specified yadi-os exit code when something found")
}

func flagsToConfig() error {
	score, err := cvs.FromLevel(rootOpts.SeverityLevel)
	if err != nil {
		return err
	}

	config.MinimumSeverity = score
	if rootOpts.FeedURI != "" {
		config.FeedURI = rootOpts.FeedURI
	}

	if _, err := cli.ListFormatter(rootOpts.Format); err != nil {
		return err
	}

	return nil
}

func rootPreRun(cmd *cobra.Command, _ []string) error {
	var root *cobra.Command
	for p := cmd; p != nil; p = p.Parent() {
		if p.Name() == "yadi-os" {
			root = p
			break
		}
	}

	if root == nil {
		return errors.New("can't find root command ('yadi-os')")
	}

	if rootOpts.ShowVersion {
		fmt.Println(config.FullVersion())
		os.Exit(0)
	}

	if err := flagsToConfig(); err != nil {
		return err
	}

	if rootOpts.Verbose {
		simplelog.SetLevel(simplelog.DebugLevel)
	} else {
		simplelog.SetLevel(simplelog.InfoLevel)
	}

	return nil
}
