package analyze

import (
	"fmt"
	"sort"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/kernel"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/manager"
)

type Analyzer struct {
	kernelFeed      *feed.KernelFeed
	packagesFeed    *feed.PackagesFeed
	fixableOnly     bool
	packagesExclude map[string]struct{}
}

func New(opts Options) (*Analyzer, error) {
	vulnerabilitiesExclude := make(map[string]struct{}, len(opts.VulnerabilitiesExclude))
	packagesExclude := make(map[string]struct{}, len(opts.PackagesExclude))

	for _, id := range opts.VulnerabilitiesExclude {
		vulnerabilitiesExclude[id] = struct{}{}
	}

	for _, pkgName := range opts.PackagesExclude {
		packagesExclude[pkgName] = struct{}{}
	}

	feedOpts := feed.Options{
		MinimumSeverity:        opts.MinimumSeverity,
		FeedURI:                opts.FeedURI,
		Fetcher:                opts.Fetcher,
		VulnerabilitiesExclude: vulnerabilitiesExclude,
	}

	result := &Analyzer{
		packagesFeed:    feed.NewPackagesFeed(feedOpts),
		kernelFeed:      feed.NewKernelFeed(feedOpts),
		fixableOnly:     opts.FixableOnly,
		packagesExclude: packagesExclude,
	}

	return result, nil
}

func (a *Analyzer) CheckKernel(linux kernel.Kernel) (CheckKernelResult, error) {
	vulnerabilities, err := a.kernelFeed.Fetch()
	if err != nil {
		return nil, xerrors.Errorf("failed to fetch vulnerabilities: %w", err)
	}

	if len(vulnerabilities) == 0 {
		return nil, nil
	}

	// TODO(buglloc): do something better
	anyVersion := fmt.Sprintf("%d.%d.*", linux.Release.Major, linux.Release.Minor)
	var result CheckKernelResult
	for _, vuln := range vulnerabilities {
		if a.fixableOnly && strings.Contains(vuln.RawVersions, anyVersion) {
			continue
		}

		if !vuln.IsAffected(linux.Release) {
			continue
		}

		result = append(result, Issue{
			ID:               vuln.ID,
			AffectedVersions: vuln.RawVersions,
			Reference:        vuln.Reference,
			CVSSScore:        vuln.CVSSScore,
			Summary:          vuln.Summary,
		})
	}
	if len(result) > 0 {
		sort.Sort(result)
	}

	return result, nil
}

func (a *Analyzer) CheckPackages(pm manager.Manager) (CheckPackagesResult, error) {
	packages, err := pm.Packages()
	if err != nil {
		return nil, xerrors.Errorf("failed to get packages: %w", err)
	}

	if len(packages) == 0 {
		return nil, nil
	}

	vulnerabilities, err := a.packagesFeed.Fetch(pm.Distributive())
	if err != nil {
		return nil, xerrors.Errorf("failed to fetch vulnerabilities: %w", err)
	}

	if len(vulnerabilities) == 0 {
		return nil, nil
	}

	distrRelease := pm.DistributiveCodename()
	result := make(CheckPackagesResult, 0)
	for _, pkg := range packages {
		var vulns Issues

		// first of all - check excludes
		var excludedPackage bool
		if _, excludedPackage = a.packagesExclude[pkg.Name]; excludedPackage {
			continue
		}

		if _, excludedPackage = a.packagesExclude[pkg.SourceName]; excludedPackage {
			continue
		}

		// then check binary pkg
		for _, vuln := range vulnerabilities.ForPackage(pkg.Name) {
			if versions, isAffected := vuln.Affected(distrRelease, pkg.Name, pkg.Version); isAffected {
				if a.fixableOnly && versions == feed.AffectedAnyVersions {
					// TODO(buglloc): ugly
					simplelog.Debug("skip unfixable issue", "pkg", pkg.Name, "id", vuln.ID)
					continue
				}

				vulns = append(vulns, Issue{
					ID:               vuln.ID,
					AffectedVersions: versions,
					Reference:        vuln.Reference,
					CVSSScore:        vuln.CVSSScore,
					Summary:          vuln.Summary,
				})
			}
		}

		// then source pkg
		if pkg.SourceName != "" {
			for _, vuln := range vulnerabilities.ForPackage(pkg.SourceName) {
				var version *debversion.Version
				if pkg.SourceVersion != nil {
					version = pkg.SourceVersion
				} else {
					version = pkg.Version
				}

				if versions, isAffected := vuln.Affected(distrRelease, pkg.SourceName, version); isAffected {
					if a.fixableOnly && versions == feed.AffectedAnyVersions {
						// TODO(buglloc): ugly
						simplelog.Debug("skip unfixable issue", "pkg", pkg.SourceName, "id", vuln.ID)
						continue
					}

					vulns = append(vulns, Issue{
						ID:               vuln.ID,
						AffectedVersions: versions,
						Reference:        vuln.Reference,
						CVSSScore:        vuln.CVSSScore,
						Summary:          vuln.Summary,
					})
				}
			}
		}

		if len(vulns) > 0 {
			sort.Sort(vulns)
			result = append(result, PackageVulnerabilities{
				Package:         pkg,
				Vulnerabilities: vulns,
			})
		}
	}
	return result, err
}

func (a *Analyzer) ListPackages(pm manager.Manager) (ListPackagesResult, error) {
	packages, err := pm.Packages()
	if err != nil {
		return nil, xerrors.Errorf("failed to get packages: %w", err)
	}

	return packages, nil
}
