package feed

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

var (
	gzipMagic                = []byte{0x1F, 0x8B, 0x08}
	ErrEcosystemNotSupported = errors.New("ecosystem is not supported")
)

type Fetcher struct {
	feedURI string
	lock    sync.Mutex
	cache   map[string]Feed
}

func NewFetcher(opts Options) *Fetcher {
	f := &Fetcher{
		feedURI: "https://yadi.yandex-team.ru/db/linux-{ecosystem}.json.gz",
		cache:   make(map[string]Feed),
	}

	if opts.FeedURI != "" {
		f.feedURI = opts.FeedURI
	}

	return f
}

func (f *Fetcher) Fetch(ecosystem string) (Feed, error) {
	f.lock.Lock()
	defer f.lock.Unlock()

	if result, ok := f.cache[ecosystem]; ok {
		return result, nil
	}

	result, err := f.fetchInternal(ecosystem)
	if err != nil {
		return nil, err
	}
	f.cache[ecosystem] = result
	return result, nil
}

func (f *Fetcher) CleanCache(language string) {
	f.lock.Lock()
	defer f.lock.Unlock()
	delete(f.cache, language)
}

func (f *Fetcher) fetchInternal(ecosystem string) (Feed, error) {
	var (
		raw []byte
		err error
	)

	feedURI := formatURI(f.feedURI, ecosystem)
	if strings.HasPrefix(feedURI, "https://") || strings.HasPrefix(feedURI, "http://") {
		raw, err = fetchRemoteDB(feedURI)
	} else if filepath.IsAbs(feedURI) {
		raw, err = fetchLocalDB(feedURI)
	} else {
		err = fmt.Errorf("unsupported feed protocol: %s", feedURI)
	}

	if err != nil {
		return nil, xerrors.Errorf("failed to fetch feed for ecosystem %q: %w", ecosystem, err)
	}

	if len(raw) >= 3 && bytes.Equal(raw[:3], gzipMagic) {
		reader, err := gzip.NewReader(bytes.NewReader(raw))
		if err != nil {
			return nil, xerrors.Errorf("failed to create new gzip reader: %w", err)
		}

		raw, err = ioutil.ReadAll(reader)
		_ = reader.Close()
		if err != nil {
			return nil, xerrors.Errorf("failed to create read gzipped feed: %w", err)
		}
	}

	var feed Feed
	err = json.Unmarshal(raw, &feed)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse feed: %w", err)
	}

	return feed, nil
}

func fetchRemoteDB(url string) ([]byte, error) {
	resp, err := yahttp.Get(url)
	if err != nil {
		return nil, err
	}

	defer yahttp.GracefulClose(resp.Body)
	if resp.StatusCode == 404 {
		return nil, ErrEcosystemNotSupported
	}
	if resp.StatusCode != 200 {
		return nil, xerrors.Errorf("unexpected feed status: %d", resp.StatusCode)
	}

	return ioutil.ReadAll(resp.Body)
}

func fetchLocalDB(path string) ([]byte, error) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		if os.IsNotExist(err) {
			return nil, ErrEcosystemNotSupported
		}

		return nil, err
	}

	return data, nil
}

func formatURI(uri, ecosystem string) string {
	return strings.NewReplacer(
		"{ecosystem}", ecosystem,
		"{lang}", ecosystem,
	).Replace(uri)
}

func prepareFetcher(opts Options) *Fetcher {
	if opts.Fetcher != nil {
		return opts.Fetcher
	}

	return NewFetcher(opts)
}
