package feed

import (
	"a.yandex-team.ru/security/libs/go/semver"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	KernelFeed struct {
		fetcher                *Fetcher
		minimumSeverity        float32
		vulnerabilitiesExclude map[string]struct{}
	}
)

func NewKernelFeed(opts Options) *KernelFeed {
	return &KernelFeed{
		fetcher:                prepareFetcher(opts),
		minimumSeverity:        opts.MinimumSeverity,
		vulnerabilitiesExclude: opts.VulnerabilitiesExclude,
	}
}

func (f *KernelFeed) Fetch() (KernelVulnerabilities, error) {
	feed, err := f.fetcher.Fetch("kernel")
	if err != nil {
		return nil, err
	}

	result := make(KernelVulnerabilities, 0, len(feed))
	for _, entry := range feed {
		if _, excludedVuln := f.vulnerabilitiesExclude[entry.ID]; excludedVuln {
			continue
		}

		if entry.CVSSScore < f.minimumSeverity {
			continue
		}

		versions, err := semver.NewConstraint(entry.VulnerableVersions)
		if err != nil {
			simplelog.Error("Failed to parse vulnerability versions: "+err.Error(),
				"module", entry.ModuleName, "versions", entry.VulnerableVersions)
			continue
		}

		result = append(
			result,
			KernelVulnerability{
				ID:          entry.ID,
				CVSSScore:   entry.CVSSScore,
				RawVersions: entry.VulnerableVersions,
				Versions:    versions,
				Summary:     entry.Summary,
				Reference:   entry.Reference,
			},
		)
	}

	return result, nil
}
