package feed

import (
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
)

const (
	AffectedAnyVersions = "*"
)

type (
	PackageVulnerability struct {
		ID          string                         `json:"id"`
		PackageName string                         `json:"package_name"`
		Versions    map[string]*debversion.Version `json:"-"`
		RawVersions string                         `json:"vulnerable_versions"`
		Summary     string                         `json:"summary"`
		Reference   string                         `json:"reference"`
		CVSSScore   float32                        `json:"cvss_score"`
	}

	PackageVulnerabilities []PackageVulnerability

	PackagesVulnerabilities map[string]PackageVulnerabilities
)

var (
	latestReleases = []string{
		"sid",
		"upstream",
	}
)

func (v *PackageVulnerability) IsAffected(osCodename, pkgName string, version *debversion.Version) bool {
	_, isAffected := v.Affected(osCodename, pkgName, version)
	return isAffected
}

func (v *PackageVulnerability) Affected(osCodename, pkgName string, version *debversion.Version) (string, bool) {
	if v.PackageName != strings.ToLower(pkgName) {
		return "", false
	}

	if v.Versions == nil {
		v.Versions = parsePackageVersions(v.RawVersions)
	}

	if vulnVer, ok := v.Versions[osCodename]; ok {
		if vulnVer == debversion.AnyVersion {
			for _, latest := range latestReleases {
				if latestVersion, ok := v.Versions[latest]; ok {
					vulnVer = latestVersion
					break
				}
			}
		}

		if vulnVer == debversion.AnyVersion {
			return AffectedAnyVersions, true
		}

		if version.LessThan(vulnVer) {
			return "<" + vulnVer.String(), true
		}
	}
	return "", false
}

func (vs PackagesVulnerabilities) ForPackage(name string) []PackageVulnerability {
	if result, found := vs[strings.ToLower(name)]; found {
		return result
	}

	return nil
}

func parsePackageVersions(versions string) map[string]*debversion.Version {
	// TODO(buglloc): simplify
	result := make(map[string]*debversion.Version)
	for _, ver := range strings.Split(versions, " ") {
		parts := strings.SplitN(ver, ":", 2)
		if len(parts) != 2 {
			simplelog.Error("failed to parse vuln version",
				"versions", versions,
				"version", ver,
				"err", "not enough parts",
			)
			continue
		}

		var parsedVersion *debversion.Version
		if parts[1] == "*" {
			parsedVersion, _ = debversion.NewVersion(parts[1])
		} else {
			var err error
			parsedVersion, err = debversion.NewVersion(parts[1][1:])
			if err != nil {
				simplelog.Error("failed to parse vuln version",
					"versions", versions,
					"version", ver,
					"err", err.Error(),
				)
				continue
			}
		}

		result[parts[0]] = parsedVersion
	}
	return result
}
