package hostctl

import (
	"fmt"
	"os"

	"a.yandex-team.ru/infra/hostctl/pkg/render"
	"a.yandex-team.ru/infra/hostctl/pkg/render/rtcutil"
	hostpb "a.yandex-team.ru/infra/hostctl/proto"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/manager"
)

type ClusterPkgs struct {
	Pkgs  []*manager.Package
	Hosts []string
}

func Render(cluster, specPath string) ([]ClusterPkgs, error) {
	pkgF, err := os.Open(specPath)
	if err != nil {
		return nil, err
	}
	defer func() { _ = pkgF.Close() }()

	st, err := render.NewLocalRenderStorage(specPath)
	if err != nil {
		return nil, fmt.Errorf("create render storage: %w", err)
	}

	cr, err := rtcutil.NewClusterRenderer(cluster)
	if err != nil {
		return nil, err
	}

	ci, err := cr.UnitFromStorage(st, render.UnitNameFromPath(specPath))
	if err != nil {
		return nil, err
	}

	out := make([]ClusterPkgs, 0, len(ci))
	for _, info := range ci {
		spec, ok := info.Spec().(*hostpb.PackageSetSpec)
		if !ok {
			return nil, fmt.Errorf("unsupported spec: %T", info.Spec())
		}

		clusterPkgs := ClusterPkgs{
			Pkgs:  make([]*manager.Package, 0, len(spec.Packages)),
			Hosts: info.Hosts(),
		}

		for _, pkg := range spec.Packages {
			parsedVersion, err := debversion.NewVersion(pkg.Version)
			if err != nil {
				simplelog.Error("failed to parse package version", "package", pkg.Name, "version", pkg.Version, "err", err)
				continue
			}

			clusterPkgs.Pkgs = append(clusterPkgs.Pkgs, &manager.Package{
				Name:       pkg.Name,
				Version:    parsedVersion,
				RawVersion: pkg.Version,
			})
		}

		out = append(out, clusterPkgs)
	}

	return out, nil
}
