package apk

import (
	"bufio"
	"os"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/lineage"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
)

const (
	StatusPath = "/lib/apk/db/installed"
)

type (
	ManagerOpts struct {
		OS   *lineage.OS
		Root string
	}

	Manager struct {
		os         *lineage.OS
		statusPath string
	}
)

var _ manager.Manager = (*Manager)(nil)

func NewManager(opts ManagerOpts) (*Manager, error) {
	mngr := &Manager{
		os:         opts.OS,
		statusPath: filepath.Join(opts.Root, StatusPath),
	}

	if !fsutils.IsFileExists(mngr.statusPath) {
		return nil, xerrors.Errorf("apk status file not found: %s", mngr.statusPath)
	}

	return mngr, nil
}

func (m *Manager) Name() string {
	return "apk"
}

func (m *Manager) Distributive() string {
	return strings.ToLower(m.os.Family)
}

func (m *Manager) DistributiveCodename() string {
	return strings.ToLower(m.os.Release)
}

func (m *Manager) Packages() ([]*manager.Package, error) {
	apkStatusFile, err := os.Open(m.statusPath)
	if err != nil {
		return nil, xerrors.Errorf("failed to open apk status file: %w", err)
	}
	defer func() { _ = apkStatusFile.Close() }()

	var result []*manager.Package
	scanner := bufio.NewScanner(apkStatusFile)
	for scanner.Scan() {
		pkg := parsePkgs(scanner)
		if pkg != nil {
			result = append(result, pkg)
		}
	}

	return result, nil
}

func parsePkgs(scanner *bufio.Scanner) (pkg *manager.Package) {
	var (
		name    string
		version string
	)

	for {
		line := strings.TrimSpace(scanner.Text())
		if line == "" {
			break
		}

		if len(line) < 2 {
			continue
		}

		switch line[:2] {
		case "P:":
			name = strings.TrimSpace(line[2:])
		case "V:":
			version = strings.TrimSpace(line[2:])
		}

		if !scanner.Scan() {
			break
		}
	}

	if name == "" || version == "" {
		return nil
	}

	parsedVersion, err := debversion.NewVersion(version)
	if err != nil {
		simplelog.Error("failed to parse package version", "package", name, "version", version, "err", err)
		return nil
	}

	return &manager.Package{
		Name:       name,
		Version:    parsedVersion,
		RawVersion: version,
	}
}
