package dpkg

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"regexp"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/lineage"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
)

const (
	StatusPath  = "/var/lib/dpkg/status"
	StatusDPath = "/var/lib/dpkg/status.d"
)

type (
	ManagerOpts struct {
		OS   *lineage.OS
		Root string
	}

	Manager struct {
		os          *lineage.OS
		statusPath  string
		statusDPath string
	}
)

var (
	sourceFiledRegexp                 = regexp.MustCompile(`^Source: (?P<name>[^\s]*)(\s*\((?P<version>.*)\))?`)
	_                 manager.Manager = (*Manager)(nil)
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	mngr := &Manager{
		os:          opts.OS,
		statusPath:  filepath.Join(opts.Root, StatusPath),
		statusDPath: filepath.Join(opts.Root, StatusDPath),
	}

	if !fsutils.IsFileExists(mngr.statusPath) && !fsutils.IsDirExists(mngr.statusDPath) {
		return nil, xerrors.Errorf("dpkg status file not found")
	}

	return mngr, nil
}

func (m *Manager) Name() string {
	return "dpkg"
}

func (m *Manager) Distributive() string {
	return strings.ToLower(m.os.Family)
}

func (m *Manager) DistributiveCodename() string {
	return strings.ToLower(m.os.Codename)
}

func (m *Manager) Packages() ([]*manager.Package, error) {
	var result []*manager.Package
	if fsutils.IsFileExists(m.statusPath) {
		pkgs, err := parseStatus(m.statusPath)
		if err != nil {
			return nil, err
		}
		result = append(result, pkgs...)
	}

	if fsutils.IsDirExists(m.statusDPath) {
		// distroless status files: https://github.com/GoogleContainerTools/distroless
		files, err := ioutil.ReadDir(m.statusDPath)
		if err != nil {
			return nil, xerrors.Errorf("failed to list status.d: %w", err)
		}

		for _, file := range files {
			if file.IsDir() {
				continue
			}

			pkgs, err := parseStatus(filepath.Join(m.statusDPath, file.Name()))
			if err != nil {
				return nil, err
			}
			result = append(result, pkgs...)
		}
	}

	return result, nil
}

func parseStatus(statusPath string) ([]*manager.Package, error) {
	dpkgStatusFile, err := os.Open(statusPath)
	if err != nil {
		return nil, xerrors.Errorf("failed to open dpkg status file %q: %w", statusPath, err)
	}
	defer func() { _ = dpkgStatusFile.Close() }()

	return ParseStatusData(dpkgStatusFile)
}
