package dpkg

import (
	"bufio"
	"io"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/manager"
)

func ParseStatusData(in io.Reader) ([]*manager.Package, error) {
	var result []*manager.Package
	scanner := bufio.NewScanner(in)
	for scanner.Scan() {
		pkg := parsePkgs(scanner)
		if pkg != nil {
			result = append(result, pkg)
		}
	}

	return result, nil
}

func parsePkgs(scanner *bufio.Scanner) *manager.Package {
	var (
		name          string
		version       string
		sourceName    string
		sourceVersion string
	)

	for {
		line := strings.TrimSpace(scanner.Text())
		if line == "" {
			break
		}

		switch {
		case strings.HasPrefix(line, "Package: "):
			name = strings.TrimSpace(line[9:])
		case strings.HasPrefix(line, "Version: "):
			version = strings.TrimSpace(line[9:])
		case strings.HasPrefix(line, "Source: "):
			// Optional source line, gives the name of the source package
			// may also include version
			parsed := sourceFiledRegexp.FindStringSubmatch(line)
			sourceName = parsed[1]
			sourceVersion = parsed[2]
		}

		if !scanner.Scan() {
			break
		}
	}

	if name == "" || version == "" {
		return nil
	}

	parsedVersion, err := debversion.NewVersion(version)
	if err != nil {
		simplelog.Error("failed to parse package version", "package", name, "version", version, "err", err)
		return nil
	}

	parsedSourceVersion := parsedVersion
	if sourceVersion == "" {
		sourceVersion = version
	} else {
		if parsed, err := debversion.NewVersion(sourceVersion); err == nil {
			parsedSourceVersion = parsed
		} else {
			sourceVersion = version
		}
	}

	return &manager.Package{
		Name:             name,
		Version:          parsedVersion,
		RawVersion:       version,
		SourceName:       sourceName,
		SourceVersion:    parsedSourceVersion,
		RawSourceVersion: sourceVersion,
	}
}
