package splunk

import (
	"errors"
	"fmt"
	"strings"

	"a.yandex-team.ru/security/libs/go/semver"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/feed"
)

type feedOrErr struct {
	f   feed.PackagesVulnerabilities
	err error
}

type analyzer struct {
	fixableOnly       bool
	skipUnsupported   bool
	pkgFeedFetcher    *feed.PackagesFeed
	kernelFeedFetcher *feed.KernelFeed
	kernelFeed        feed.KernelVulnerabilities
	feeds             map[string]feedOrErr
}

func NewAnalyzer(opts options) *analyzer {
	return &analyzer{
		pkgFeedFetcher: feed.NewPackagesFeed(feed.Options{
			FeedURI:         opts.feedURI,
			MinimumSeverity: opts.minSeverity,
		}),
		kernelFeedFetcher: feed.NewKernelFeed(feed.Options{
			FeedURI:         opts.feedURI,
			MinimumSeverity: opts.minSeverity,
		}),
		fixableOnly:     opts.fixableOnly,
		skipUnsupported: opts.skipUnsupported,
		feeds:           make(map[string]feedOrErr, 1),
	}
}

func (a *analyzer) CheckPackageEntry(e PackageEntry) ([]Issue, error) {
	vulnerabilities, err := a.pkgsFeed(e.OsPlatform)
	if err != nil {
		if errors.Is(err, feed.ErrEcosystemNotSupported) {
			return nil, nil
		}

		return nil, err
	}

	if len(vulnerabilities) == 0 {
		return nil, nil
	}

	if e.PkgVer == "" {
		return nil, errors.New("no pkg_ver provided")
	}

	pkgVer, err := debversion.NewVersion(e.PkgVer)
	if err != nil {
		return nil, fmt.Errorf("failed to parse package version (%q): %q", e.PkgVer, err)
	}

	var out []Issue
	for _, vuln := range vulnerabilities.ForPackage(e.PkgName) {
		if versions, isAffected := vuln.Affected(e.OsCodename, e.PkgName, pkgVer); isAffected {
			if a.fixableOnly && versions == feed.AffectedAnyVersions {
				continue
			}

			out = append(out, Issue{
				ID:               vuln.ID,
				AffectedVersions: versions,
				Reference:        vuln.Reference,
				CVSSScore:        vuln.CVSSScore,
				Summary:          vuln.Summary,
			})
		}
	}

	return out, nil
}

func (a *analyzer) CheckKernelEntry(e KernelEntry) ([]Issue, error) {
	vulnerabilities, err := a.kernelsFeed()
	if err != nil {
		return nil, err
	}

	if len(vulnerabilities) == 0 {
		return nil, nil
	}

	rawKernelVersion := e.KernelVersion

	// TODO(buglloc): fix it, this is bullshit
	if idx := strings.Index(rawKernelVersion, "-"); idx > 0 {
		rawKernelVersion = rawKernelVersion[:idx]
	}

	if rawKernelVersion == "" {
		return nil, errors.New("no kernel_version provided")
	}

	kernelVer, err := semver.NewVersion(rawKernelVersion)
	if err != nil {
		return nil, fmt.Errorf("failed to parse package version (%q): %q", e.KernelVersion, err)
	}

	// TODO(buglloc): do something better
	anyVersion := fmt.Sprintf("%d.%d.*", kernelVer.Major, kernelVer.Minor)
	var out []Issue
	for _, vuln := range vulnerabilities {
		if a.fixableOnly && strings.Contains(vuln.RawVersions, anyVersion) {
			continue
		}

		if !vuln.IsAffected(kernelVer) {
			continue
		}

		out = append(out, Issue{
			ID:               vuln.ID,
			AffectedVersions: vuln.RawVersions,
			Reference:        vuln.Reference,
			CVSSScore:        vuln.CVSSScore,
			Summary:          vuln.Summary,
		})

	}

	return out, nil
}

func (a *analyzer) pkgsFeed(distr string) (feed.PackagesVulnerabilities, error) {
	if out, ok := a.feeds[distr]; ok {
		return out.f, out.err
	}

	out, err := a.pkgFeedFetcher.Fetch(distr)
	a.feeds[distr] = feedOrErr{
		f:   out,
		err: err,
	}
	return out, err
}

func (a *analyzer) kernelsFeed() (feed.KernelVulnerabilities, error) {
	if len(a.kernelFeed) > 0 {
		return a.kernelFeed, nil
	}

	out, err := a.kernelFeedFetcher.Fetch()
	if err != nil {
		return nil, err
	}

	a.kernelFeed = out
	return out, err
}
