package splunk

import (
	"errors"
	"io"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/config"
)

func ProcessPackagesLookup(in io.Reader, out io.Writer, opts ...Option) error {
	opt := options{
		fixableOnly:     true,
		skipUnsupported: true,
		feedURI:         config.FeedURI,
		minSeverity:     cvs.MediumSeverity,
	}

	for _, o := range opts {
		o(&opt)
	}

	checker := NewAnalyzer(opt)
	inCSV := NewCSVReader(in)
	csvHeaders, err := inCSV.Header(PackageFields...)
	if err != nil {
		if errors.Is(err, io.EOF) {
			// no headers - no data :)
			return nil
		}

		return err
	}

	outCSV := NewCSVWriter(out, csvHeaders)
	if err := outCSV.WriteHeader(); err != nil {
		return err
	}

	for inCSV.More() {
		entry, err := inCSV.DecodePackageEntry()
		if err != nil {
			simplelog.Error("failed to decode csv entry", "entry", entry, "err", err)
			continue
		}

		entry.Issues, err = checker.CheckPackageEntry(entry)
		if err != nil {
			simplelog.Error("failed to analyze entry", "entry", entry, "err", err)
			continue
		}

		err = outCSV.Write(entry.BaseEntry)
		if err != nil {
			simplelog.Error("failed to write entry", "entry", entry, "err", err)
		}
	}

	return outCSV.Close()
}

func ProcessKernelLookup(in io.Reader, out io.Writer, opts ...Option) error {
	opt := options{
		fixableOnly: true,
		feedURI:     config.FeedURI,
		minSeverity: cvs.MediumSeverity,
	}

	for _, o := range opts {
		o(&opt)
	}

	checker := NewAnalyzer(opt)
	inCSV := NewCSVReader(in)
	csvHeaders, err := inCSV.Header(KernelFields...)
	if err != nil {
		if errors.Is(err, io.EOF) {
			// no headers - no data :)
			return nil
		}

		return err
	}

	outCSV := NewCSVWriter(out, csvHeaders)
	if err := outCSV.WriteHeader(); err != nil {
		return err
	}

	for inCSV.More() {
		entry, err := inCSV.DecodeKernelEntry()
		if err != nil {
			simplelog.Error("failed to decode csv entry", "entry", entry, "err", err)
			continue
		}

		entry.Issues, err = checker.CheckKernelEntry(entry)
		if err != nil {
			simplelog.Error("failed to analyze entry", "entry", entry, "err", err)
			continue
		}

		err = outCSV.Write(entry.BaseEntry)
		if err != nil {
			simplelog.Error("failed to write entry", "entry", entry, "err", err)
		}
	}

	return outCSV.Close()
}
