package splunk

import (
	"encoding/csv"
	"errors"
	"fmt"
	"io"
)

var PackageFields = []string{
	FieldOsPlatform,
	FieldOsCodename,
	FieldPkgName,
	FieldPkgVer,
}

var KernelFields = []string{
	FieldKernelVersion,
}

type CSVReader struct {
	r              *csv.Reader
	err            error
	entries        []string
	headerParsed   bool
	headers        []string
	ignoredIndexes map[int]struct{}
	entryIndexes   map[string]int
}

func NewCSVReader(r io.Reader) *CSVReader {
	return &CSVReader{
		r: csv.NewReader(r),
	}
}

func (r *CSVReader) Header(requiredFields ...string) ([]string, error) {
	if r.headerParsed {
		return r.headers, nil
	}

	headers, err := r.r.Read()
	if err != nil {
		return nil, fmt.Errorf("failed to parse csv header: %w", err)
	}

	ignoredIndexes := make(map[int]struct{})
	for _, igH := range outHeaders {
		for idx, h := range headers {
			if h != igH {
				continue
			}

			ignoredIndexes[idx] = struct{}{}
		}
	}
	headers = filterStringIndexes(headers, ignoredIndexes)

	entryIndexes := make(map[string]int, len(requiredFields))
	for _, inH := range requiredFields {
		found := false
		for idx, h := range headers {
			if h != inH {
				continue
			}

			entryIndexes[inH] = idx
			found = true
			break
		}

		if !found {
			return nil, fmt.Errorf("required header %q not found", inH)
		}
	}

	r.headerParsed = true
	r.headers = headers
	r.ignoredIndexes = ignoredIndexes
	r.entryIndexes = entryIndexes
	return r.headers, nil
}

func (r *CSVReader) More() bool {
	if !r.headerParsed {
		r.err = errors.New("you must call .Header first")
		return false
	}

	entries, err := r.r.Read()
	switch err {
	case nil:
		// pass
	case io.EOF:
		// reached end of the file
		return false
	default:
		r.err = err
		return true
	}

	r.entries = filterStringIndexes(entries, r.ignoredIndexes)
	return true
}

func (r *CSVReader) DecodePackageEntry() (PackageEntry, error) {
	if r.err != nil {
		return PackageEntry{}, r.err
	}

	out := PackageEntry{
		BaseEntry: BaseEntry{
			allFields: r.entries,
		},
	}

	var err error
	out.OsPlatform, err = r.parseField(FieldOsPlatform)
	if err != nil {
		return out, err
	}

	out.OsCodename, err = r.parseField(FieldOsCodename)
	if err != nil {
		return out, err
	}

	out.PkgName, err = r.parseField(FieldPkgName)
	if err != nil {
		return out, err
	}

	out.PkgVer, err = r.parseField(FieldPkgVer)
	if err != nil {
		return out, err
	}

	return out, nil
}

func (r *CSVReader) DecodeKernelEntry() (KernelEntry, error) {
	out := KernelEntry{
		BaseEntry: BaseEntry{
			allFields: r.entries,
		},
	}

	var err error
	out.KernelVersion, err = r.parseField(FieldKernelVersion)
	if err != nil {
		return out, err
	}

	return out, nil
}

func (r *CSVReader) parseField(name string) (string, error) {
	idx, ok := r.entryIndexes[name]
	if !ok {
		return "", fmt.Errorf("failed to parse field %q: no entry index found", name)
	}

	if idx >= len(r.entries) {
		return "", fmt.Errorf("failed to parse field %q: invalid entries: %d (idx) >= %d (entries)", name, idx, len(r.entries))
	}

	return r.entries[idx], nil
}
