package splunk

import (
	"encoding/csv"
	"fmt"
	"io"
	"strings"

	"a.yandex-team.ru/security/yadi/libs/cvs"
)

var outHeaders = []string{
	FieldVulnerable,
	FieldVulnID,
	FieldVulnVersions,
	FieldVulnSummary,
	FieldVulnReference,
	FieldVulnSeverity,
}

type CSVWriter struct {
	w            *csv.Writer
	headerWrited bool
	headers      []string
}

func NewCSVWriter(w io.Writer, headers []string) *CSVWriter {
	return &CSVWriter{
		w:       csv.NewWriter(w),
		headers: append(headers, outHeaders...),
	}
}

func (w *CSVWriter) WriteHeader() error {
	if w.headerWrited {
		return nil
	}

	if err := w.w.Write(w.headers); err != nil {
		return fmt.Errorf("failed to write header: %w", err)
	}

	w.headerWrited = true
	return nil
}

func (w *CSVWriter) Write(e BaseEntry) error {
	if err := w.WriteHeader(); err != nil {
		return err
	}

	if len(e.Issues) == 0 {
		return w.w.Write(append(e.allFields, "false", "", "", "", "", ""))
	}

	for _, issue := range e.Issues {
		if err := w.w.Write(append(e.allFields, issueToCSV(issue)...)); err != nil {
			return err
		}
	}

	return nil
}

func (w *CSVWriter) Close() error {
	w.w.Flush()
	return w.w.Error()
}

func issueToCSV(issue Issue) []string {
	return []string{
		"true",
		issue.ID,
		issue.AffectedVersions,
		issue.Summary,
		issue.Reference,
		strings.ToLower(cvs.ToSeverity(issue.CVSSScore)),
	}
}
