package integration

import (
	"bytes"
	"context"
	"encoding/json"
	"io/ioutil"
	"sort"
	"strings"
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/stretchr/testify/require"

	"a.yandex-team.ru/library/go/test/yatest"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/jsonout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/writer"
)

type (
	Module struct {
		Path         string     `json:"path,omitempty"`
		PackageName  string     `json:"package_name"`
		License      string     `json:"license,omitempty"`
		Language     string     `json:"language,omitempty"`
		Version      string     `json:"version"`
		From         string     `json:"from,omitempty"`
		Dependencies ModuleList `json:"dependencies,omitempty"`
	}

	AnalyzeResult struct {
		Issues map[string]analyze.IssueList `json:"issues"`
	}

	ModuleList []*Module

	AnalyzeTable struct {
		Manager    manager.PackageManager
		ResultPath string
		RootPath   string
	}

	AnalyzePkgTable struct {
		Manager manager.PackageManager
		Package struct {
			Name, Version string
		}
		ResultPath string
	}

	WalkTable struct {
		Manager    manager.PackageManager
		ResultPath string
	}
)

var ArcadiaRoot = yatest.SourcePath(".")

func (l ModuleList) Sort() {
	sort.Slice(l, func(i, j int) bool {
		return l[i].PackageName+l[i].Version < l[j].PackageName+l[j].Version
	})

	for i := range l {
		l[i].Dependencies.Sort()
	}
}

func issueLess(i, j analyze.Issue) bool {
	switch {
	case i.PackageName != j.PackageName:
		return i.PackageName < j.PackageName
	case i.Version != j.Version:
		return i.Version < j.Version
	case i.Vulnerability.ID != j.Vulnerability.ID:
		return i.Vulnerability.ID < j.Vulnerability.ID
	case len(i.Path) != len(j.Path):
		return len(i.Path) < len(j.Path)
	default:
		return strings.Join(i.Path, "-") < strings.Join(j.Path, "-")
	}
}

// For testing with Analyze you should prepare canonical result of found issues. For example:
//  yadi test -l --no-suggest -f json --feed <path to fixed feed, eg. feed from sbr://1614842931> <path to target> | jq -c '.[] | {issues: {(.path):.issues}}' > issues.json
// Don't forget cut your arcadia root from "path" in the result file.
func Analyze(t *testing.T, analyzer analyze.Analyzer, table AnalyzeTable) {
	var expected AnalyzeResult

	// to detect path to module correctly
	arcadiaRoot := table.RootPath
	if arcadiaRoot == "" {
		// autodetect it
		arcadiaRoot = ArcadiaRoot
	}

	result, err := analyzer.Analyze(context.Background(), analyze.Request{
		PackageManager: table.Manager,
	})
	require.NoError(t, err)

	expectedData, err := ioutil.ReadFile(table.ResultPath)
	require.NoError(t, err)

	err = json.Unmarshal(expectedData, &expected)
	require.NoError(t, err)

	cmpOpts := cmp.Options{
		cmpopts.IgnoreFields(analyze.Issue{}, "Vulnerability.Versions", "Suggest"),
		cmpopts.SortSlices(issueLess),
	}

	for root, resultIssues := range result.Issues {
		root = strings.TrimPrefix(root, arcadiaRoot)
		t.Run(root, func(t *testing.T) {
			expectedIssues, ok := expected.Issues[root]
			require.True(t, ok, "cannot find root '%s', available: %s", root, expected.Issues)
			require.Len(t, resultIssues, len(expectedIssues))
			require.True(t, cmp.Equal(expectedIssues, resultIssues, cmpOpts...), cmp.Diff(expectedIssues, resultIssues, cmpOpts...))
		})
	}
}

func AnalyzePkg(t *testing.T, analyzer analyze.Analyzer, table AnalyzePkgTable) {
	resultIssues, err := analyzer.AnalyzePkg(context.Background(), analyze.PkgRequest{
		PackageManager: table.Manager,
		PackageName:    table.Package.Name,
		PackageVersion: table.Package.Version,
	})
	require.NoError(t, err)

	expectedData, err := ioutil.ReadFile(table.ResultPath)
	require.NoError(t, err)

	var expectedIssues analyze.IssueList
	err = json.Unmarshal(expectedData, &expectedIssues)
	require.NoError(t, err)

	require.Len(t, resultIssues, len(expectedIssues))

	cmpOpts := cmp.Options{
		cmpopts.IgnoreFields(analyze.Issue{}, "Vulnerability.Versions", "Suggest"),
		cmpopts.SortSlices(issueLess),
	}

	require.True(t, cmp.Equal(expectedIssues, resultIssues, cmpOpts...), cmp.Diff(expectedIssues, resultIssues))
}

func Walk(t *testing.T, analyzer analyze.Analyzer, table WalkTable) {
	var expected ModuleList
	var result ModuleList

	expectedData, err := ioutil.ReadFile(table.ResultPath)
	require.NoError(t, err)

	err = json.Unmarshal(expectedData, &expected)
	require.NoError(t, err)

	out := bytes.NewBuffer(nil)
	consumer := jsonout.NewListOutput(
		jsonout.WithWriter(
			writer.NewWriter(
				writer.WithBuf(out),
			),
		),
	)

	err = analyzer.Walk(context.Background(), analyze.WalkRequest{
		PackageManager: table.Manager,
		Consumer:       consumer,
	})
	require.NoError(t, err)

	err = consumer.Close()
	require.NoError(t, err)

	err = json.Unmarshal(out.Bytes(), &result)
	require.NoError(t, err)

	expected.Sort()
	result.Sort()

	cmpOpts := cmp.Options{
		cmpopts.IgnoreFields(Module{}, "Path"),
	}
	require.True(t, cmp.Equal(expected, result, cmpOpts...), cmp.Diff(expected, result))
}
