package cli

import (
	"fmt"
	"os"
	"strings"

	"golang.org/x/crypto/ssh/terminal"

	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi/internal/service"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/consoleout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/genericout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/jsonout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/startrekout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/teamcityout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/textout"
)

const (
	TextFormatterName     = "text"
	ConsoleFormatterName  = "console"
	JSONFormatterName     = "json"
	StFormatterName       = "st"
	StFullFormatterName   = "startrek"
	TeamcityFormatterName = "teamcity"
	GenericFormatterName  = "generic"
)

type ListOptions struct {
	Format, Project string
	Languages       []string
}

func FetcherFlags(localFlag, remoteFlag bool) manager.ResolveMode {
	if !(localFlag || remoteFlag) {
		// Use both by default
		return manager.ResolveLocal | manager.ResolveRemote
	}

	var result manager.ResolveMode
	if localFlag {
		result |= manager.ResolveLocal
	}
	if remoteFlag {
		result |= manager.ResolveRemote
	}
	return result
}

func CheckFormatter(format string) error {
	formatters := map[string]struct{}{
		TextFormatterName:     {},
		ConsoleFormatterName:  {},
		JSONFormatterName:     {},
		StFormatterName:       {},
		StFullFormatterName:   {},
		TeamcityFormatterName: {},
		GenericFormatterName:  {},
	}
	if _, ok := formatters[format]; !ok {
		return fmt.Errorf("unknown format: %s", format)
	}
	return nil
}

func IssuesFormatter(format string) (outer.IssueOutput, error) {
	switch format {
	case TextFormatterName:
		return textout.NewIssueOutput(), nil
	case ConsoleFormatterName:
		return consoleout.NewIssueOutput(), nil
	case JSONFormatterName:
		return jsonout.NewIssueOutput(), nil
	case StFormatterName, StFullFormatterName:
		return startrekout.NewIssueOutput(), nil
	case TeamcityFormatterName:
		return teamcityout.NewIssueOutput(), nil
	case GenericFormatterName:
		return genericout.NewIssueOutput(), fmt.Errorf("generic format is supported only for list command")
	default:
		return nil, fmt.Errorf(
			"unrecognized format %q, must be one of: text, console, json, startrek or teamcity",
			format,
		)
	}
}

func DetectDefaultFormatter() string {
	switch {
	case IsRunningUnderTeamcity():
		// If we started on teamcity agent - use teamcity formatter by default
		return TeamcityFormatterName
	case terminal.IsTerminal(int(os.Stdout.Fd())):
		return ConsoleFormatterName
	default:
		return TextFormatterName
	}
}

func ListFormatter(opts ListOptions) (outer.ListOutput, error) {
	switch opts.Format {
	case TextFormatterName:
		return textout.NewListOutput(), nil
	case ConsoleFormatterName:
		return textout.NewListOutput(), nil
	case JSONFormatterName:
		return jsonout.NewListOutput(), nil
	case StFormatterName, StFullFormatterName:
		return textout.NewListOutput(), nil
	case TeamcityFormatterName:
		return textout.NewListOutput(), nil
	case GenericFormatterName:
		if len(opts.Languages) != 1 {
			return nil, fmt.Errorf("please, specify the only one language for generic format: %v", opts.Languages)
		}
		return genericout.NewListOutput(
			genericout.WithProjectName(opts.Project),
			genericout.WithLanguage(opts.Languages[0]),
		), nil
	default:
		return nil, fmt.Errorf(
			"unrecognized format %q, must be one of: text, console, json, startrek, teamcity or generic",
			opts.Format,
		)
	}
}

func DumpProjectInfo(project *service.ProjectInfo) string {
	result := ""
	result += fmt.Sprintf("Service: %s\n", project.Service)
	result += fmt.Sprintf("Name : %s\n", project.Name)
	result += fmt.Sprintf("Additional project Owners: %s\n", project.Owners)
	result += fmt.Sprintf("Clone Uri: %s\n", project.RepoURI)
	result += fmt.Sprintf("Minimum severity: %s\n", cvs.ToSeverity(project.Severity))
	if len(project.Targets) > 0 {
		result += "Targets: \n"
		for _, target := range project.Targets {
			result += fmt.Sprintf("  - %s\n", target)
		}
	}

	return result
}

func SeverityToLevel(severity string) int {
	switch strings.ToLower(severity) {
	case "info":
		return 0
	case "low":
		return 1
	case "medium":
		return 2
	case "high":
		return 3
	case "critical":
		return 4
	default:
		return 2
	}
}

func IsRunningUnderTeamcity() bool {
	/*
		https://confluence.jetbrains.com/display/TCD10/Predefined+Build+Parameters#PredefinedBuildParameters-ServerBuildProperties
		TEAMCITY_VERSION: The version of TeamCity server. This property can be used to determine the build is run within TeamCity.
	*/

	return os.Getenv("TEAMCITY_VERSION") != ""
}
