package cli

import (
	"context"
	"fmt"
	"io/fs"
	"os"
	"path/filepath"
	"sort"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/internal/config"
	"a.yandex-team.ru/security/yadi/yadi/internal/results"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/bazelmaven"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/generic"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/gomod"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/maven"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npmshrinkwrap"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pkglock"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/yarn"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
)

var defExcludeDirs = []string{
	".git",
	"node_modules",
	"vendor",
	".npm",
	"env",
	".env",
	"venv",
	".venv",
	"eggs",
}

type (
	DepsOptions struct {
		Targets     []string
		FuzzySearch bool
		ResolveMode manager.ResolveMode
		IssueOutput outer.IssueOutput
		ListOutput  outer.ListOutput
	}
)

func newAnalyzer() analyze.Analyzer {
	return analyze.NewAnalyzer(
		analyze.WithFeedOptions(feed.Options{
			MinimumSeverity: config.MinimumSeverity,
			FeedURI:         config.FeedURI,
		}),
		analyze.WithStatsTracking(false),
		analyze.WithVulnerabilityExcludes(config.SkipIssues),
		analyze.WithSuggest(config.SuggestUpdate),
	)
}

func AnalyzeTargets(ctx context.Context, opts DepsOptions) (bool, error) {
	analyzer := newAnalyzer()
	result := make([]results.Analyze, 0, len(opts.Targets))
	for _, target := range opts.Targets {
		simplelog.Info("target: " + target)
		pm, err := NewPackageManager(target, opts.ResolveMode, opts.FuzzySearch)
		if err != nil {
			simplelog.Error("failed to create new package manager", "target", target, "err", err)
			continue
		}

		analyzeResult, err := analyzer.Analyze(ctx, analyze.Request{
			PackageManager: pm,
		})
		if err != nil {
			simplelog.Error("failed to analyze package", "target", target, "pm", pm.Name(), "err", err)
			continue
		}

		if len(analyzeResult.Issues) > 0 {
			for path, issues := range analyzeResult.Issues {
				if len(issues) == 0 {
					continue
				}

				sort.Sort(issues)
				result = append(result, results.Analyze{
					Path:   path,
					Issues: issues,
				})
			}
		}
	}

	opts.IssueOutput.WriteIssues(result)
	return len(result) > 0, nil
}

func ListTargets(ctx context.Context, opts DepsOptions) error {
	analyzer := newAnalyzer()
	for _, target := range opts.Targets {
		simplelog.Info("target: " + target)
		pm, err := NewPackageManager(target, opts.ResolveMode, opts.FuzzySearch)
		if err != nil {
			simplelog.Error("failed to create new package manager", "target", target)
			continue
		}

		err = analyzer.Walk(ctx, analyze.WalkRequest{
			PackageManager: pm,
			Consumer:       opts.ListOutput,
		})

		if err != nil {
			simplelog.Error("failed to list packages", "target", target, "pm", pm.Name(), "err", err)
			continue
		}
	}

	return nil
}

func ValidateTargets(targets []string, explicit bool) []string {
	result := make([]string, 0)
	for _, target := range targets {
		candidates, err := filepath.Glob(target)
		if err != nil {
			simplelog.Error("failed to find target: " + err.Error())
			continue
		}

		for _, candidate := range candidates {
			baseName := filepath.Base(candidate)
			dirName := filepath.Base(filepath.Dir(candidate))

			if !explicit && isPythonFile(dirName, baseName) {
				if isPythonDev(candidate) {
					// Skip python dev dependencies
					simplelog.Debug("skip target: "+target, "reason", "dev requirements")
					continue
				}
			}

			path, err := filepath.Abs(candidate)
			if err != nil {
				simplelog.Error("failed to find target: " + err.Error())
				continue
			}

			stat, err := os.Stat(path)
			if err != nil {
				simplelog.Error("failed to find target: " + err.Error())
				continue
			}

			if !fsutils.IsRegular(stat) {
				simplelog.Error("skip not regular file: " + path)
				continue
			}

			result = append(result, path)
		}
	}
	return result
}

func FindTargets(dir string) []string {
	dir, err := filepath.Abs(dir)
	if err != nil {
		simplelog.Error("failed to process directory: " + err.Error())
		return nil
	}

	simplelog.Info("search projects dependencies files in: " + dir)
	var result []string
	err = filepath.WalkDir(dir, func(path string, de fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		baseName := de.Name()
		dirName := filepath.Base(filepath.Dir(path))

		for _, excl := range defExcludeDirs {
			if excl == baseName && de.IsDir() {
				simplelog.Debug("skip paht: "+path, "reason", "excluded")
				return filepath.SkipDir
			}
		}

		accepted := false
		skipDir := false
		switch {
		case isJavascriptFile(baseName):
			accepted = true
			pj, err := npm.ParsePackageJSONFile(path, true)
			if err != nil {
				simplelog.Debug("failed to parse package.json", "path", path, "err", err.Error())
				return nil
			}
			if len(pj.Workspaces) > 0 {
				// used Node.js workspaces, so we must skip any nested packages
				skipDir = de.IsDir()
			}
		case isBazelMavenFile(baseName):
			accepted = true
		case isGolangFile(baseName):
			accepted = true
		case isJavaFile(baseName):
			accepted = true
		case isPythonFile(dirName, baseName):
			// Exclude python dev dependencies
			accepted = !isPythonDev(baseName)
			if !accepted {
				simplelog.Debug("skip path: "+path, "reason", "dev requirements")
			}
		}

		if !accepted {
			// Not interesting
			return nil
		}

		stat, err := de.Info()
		if err != nil {
			simplelog.Debug("skip path: "+path, "reason", err)
			return nil
		}

		if !fsutils.IsRegular(stat) {
			simplelog.Debug("skip path: "+path, "reason", "not regular file")
			return nil
		}

		if stat.Size() > 1024*1024*100 {
			simplelog.Debug("skip path: "+path, "reason", "exceeds maximum file size")
			return nil
		}

		result = append(result, path)
		if skipDir {
			return filepath.SkipDir
		}
		return nil
	})

	if err != nil {
		simplelog.Warn("failed to walk directory tree", "err", err)
	}

	return result
}

func NewPackageManager(target string, mode manager.ResolveMode, fuzzySearch bool) (manager.PackageManager, error) {
	dir, fileName := filepath.Split(target)
	dirName := filepath.Base(dir)

	if (!config.AllowJs && !config.AllowGo && !config.AllowJava && !config.AllowBazelMaven) || isPythonFile(dirName, fileName) {
		// If we allow only Python projects - treat any files as pip requirements
		return pip.NewManager(pip.ManagerOpts{
			TargetPath:       target,
			WithDev:          config.CheckDevDependencies,
			ResolveMode:      mode,
			WithoutPypiCache: config.NoPypiCache,
		})
	}

	// Otherwise find needed pkg manager
	switch fileName {
	case "npm-shrinkwrap.json":
		return npmshrinkwrap.NewManager(npmshrinkwrap.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})
	case "yarn.lock":
		return yarn.NewManager(yarn.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})
	case "package-lock.json":
		return pkglock.NewManager(pkglock.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})
	case "go.mod":
		return gomod.NewManager(gomod.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})
	case "pom.xml":
		return maven.NewManager(maven.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})
	case "yadi.json":
		return generic.NewManager(generic.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})
	case "maven_install.json":
		return bazelmaven.NewManager(bazelmaven.ManagerOpts{
			TargetPath: target,
			WithDev:    config.CheckDevDependencies,
		})

	}

	if !isJavascriptFile(fileName) {
		if !config.AllowPython {
			// If we allow only Node.js projects - treat any files as NPM package
			return npm.NewManager(npm.ManagerOpts{
				TargetPath:  target,
				WithDev:     config.CheckDevDependencies,
				ResolveMode: mode,
			})
		}
		return nil, fmt.Errorf("unknown project language: %s", target)
	}

	if fuzzySearch {
		yarnPath := filepath.Join(dir, "yarn.lock")
		if fsutils.IsFileExists(yarnPath) {
			simplelog.Info("switch to Yarn analyze: " + yarnPath)
			result, err := yarn.NewManager(yarn.ManagerOpts{
				TargetPath: yarnPath,
				WithDev:    config.CheckDevDependencies,
			})

			if err == nil {
				return result, nil
			}

			simplelog.Error("failed to analyze yarn.lock: " + err.Error())
			simplelog.Info("continue in normal mode")
		}

		pkglockPath := filepath.Join(dir, "package-lock.json")
		if fsutils.IsFileExists(pkglockPath) {
			simplelog.Info("switch to package-lock.json analyze: " + pkglockPath)
			result, err := pkglock.NewManager(pkglock.ManagerOpts{
				TargetPath: pkglockPath,
				WithDev:    config.CheckDevDependencies,
			})

			if err == nil {
				return result, nil
			}

			simplelog.Error("failed to analyze package-lock.json: " + err.Error())
			simplelog.Info("continue in normal mode")
		}

		shrinkwrapPath := filepath.Join(dir, "npm-shrinkwrap.json")
		if fsutils.IsFileExists(shrinkwrapPath) {
			simplelog.Info("switch to NPM Shrinkwrap analyze: " + shrinkwrapPath)
			result, err := npmshrinkwrap.NewManager(npmshrinkwrap.ManagerOpts{
				TargetPath: shrinkwrapPath,
				WithDev:    config.CheckDevDependencies,
			})

			if err == nil {
				return result, nil
			}

			simplelog.Error("failed to analyze npm-shrinkwrap.json", "err", err)
			simplelog.Info("continue in normal mode")
		}
	}

	return npm.NewManager(npm.ManagerOpts{
		TargetPath:  target,
		WithDev:     config.CheckDevDependencies,
		ResolveMode: mode,
	})
}

func isPythonFile(dirname, filename string) bool {
	if !config.AllowPython {
		return false
	}

	if dirname != "requirements" && !strings.HasPrefix(filename, "req") {
		return false
	}

	if !strings.HasSuffix(filename, ".txt") && !strings.HasSuffix(filename, ".pip") {
		return false
	}

	return true
}

func isPythonDev(filename string) bool {
	if config.CheckDevDependencies {
		// Allow dev dependencies if user want it :)
		return false
	}

	if strings.Contains(filename, "dev") {
		return true
	}

	return strings.Contains(filename, "test")
}

func isJavascriptFile(filename string) bool {
	if !config.AllowJs {
		return false
	}

	return filename == "package.json"
}

func isBazelMavenFile(filename string) bool {
	if !config.AllowBazelMaven {
		return false
	}

	return filename == "maven_install.json"
}

func isGolangFile(filename string) bool {
	if !config.AllowGo {
		return false
	}

	switch filename {
	case "go.mod":
		return true
	}
	return false
}

func isJavaFile(filename string) bool {
	if !config.AllowJava {
		return false
	}

	return filename == "pom.xml"
}
