package commands

import (
	"errors"
	"fmt"
	"os"
	"path/filepath"
	"regexp"
	"strings"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/gitconfig"
	"a.yandex-team.ru/security/libs/go/prompt"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi/internal/service"
)

var initCmd = &cobra.Command{
	Use:   "init",
	Short: "Register new project in the Yadi service",
	Long:  "Register new project in the Yadi service",
	RunE:  runInitCmd,
}

func init() {
	RootCmd.AddCommand(initCmd)
}

func runInitCmd(cmd *cobra.Command, args []string) error {
	targets := findInitTargets()
	project, err := askProjectInfo(targets)
	if err != nil {
		fmt.Println("Aborted.")
		return nil
	}

	info, err := service.NewProject(project)
	if err != nil {
		fmt.Printf("Failed to register project: %s\n", err.Error())
		return nil
	}

	fmt.Printf("Now open this url in the browser to complete registration: %s\n", info.ConfirmURI)
	for {
		time.Sleep(3 * time.Second)
		active, err := service.ValidateProjectActive(project.Service, project.Name)
		if err != nil {
			fmt.Printf("Failed to fetch project: %s\n", err.Error())
			return nil
		}

		if active {
			break
		}
	}

	path, _ := filepath.Abs(".yadi.toml")
	err = cli.WriteConfig(path, project)
	if err != nil {
		fmt.Printf("Failed to write project file '.yadi.toml': %s", err.Error())
		return nil
	}

	fmt.Printf("Project info successful writed into: %s\n", path)
	return nil
}

func askProjectInfo(targets []string) (res *service.ProjectInfo, err error) {
	ui, err := prompt.NewUI(prompt.UIOptions{})
	if err != nil {
		return nil, err
	}

	res = &service.ProjectInfo{Targets: targets}
	var abcID string
	_, err = ui.Ask(
		prompt.AskOptions{
			Query: "ABC service slug or url",
			ValidateFunc: func(input string) error {
				abcID = normalizeAbcService(input)
				if abcID == "" {
					return errors.New("failed to normalize ABC service slug")
				}

				if err := service.ValidateAbcService(abcID); err != nil {
					return err
				}

				return nil
			},
		})
	if err != nil {
		return
	}
	res.Service = abcID

	name, err := ui.Ask(
		prompt.AskOptions{
			Query:   "Project name",
			Default: guessProjectName(res.Service),
			ValidateFunc: func(input string) error {
				if err := service.ValidateProject(res.Service, input); err != nil {
					return err
				}

				return nil
			},
		})
	if err != nil {
		return
	}
	res.Name = name

	repoURI, err := ui.Ask(
		prompt.AskOptions{
			Query:   "Repository checkout URI",
			Default: guessRepoURI(),
		})
	if err != nil {
		return
	}
	res.RepoURI = repoURI

	var severity = cvs.HighSeverity
	_, err = ui.Ask(
		prompt.AskOptions{
			Query:   "Reporting severity level (low, medium, high)",
			Default: "high",
			ValidateFunc: func(input string) error {
				severity, err = cvs.FromSeverity(input)
				if err != nil {
					return errors.New("unknown severity level")
				}
				return nil
			},
		})
	if err != nil {
		return
	}
	res.Severity = severity

	owners, err := ui.Ask(
		prompt.AskOptions{
			Query:   "Additional project owners (space separated list)",
			Default: "none",
		})
	if err != nil {
		return
	}

	if owners != "owners" {
		res.Owners = owners
	}

	fmt.Print("\nAbout to register the new project:\n\n")
	fmt.Println(cli.DumpProjectInfo(res))

	ok := false
	_, err = ui.Ask(
		prompt.AskOptions{
			Query:   "Is this ok?",
			Default: "Yes",
			ValidateFunc: func(input string) error {
				rawOk := strings.ToLower(input)
				switch rawOk {
				case "y", "yes":
					ok = true
				case "n", "no":
					ok = false
				default:
					return errors.New("must be Yes or No")
				}
				return nil
			},
		})
	if err != nil {
		return
	}

	if !ok {
		err = errors.New("not OK :(")
		return
	}
	return res, nil
}

func guessProjectName(serviceID string) string {
	dir, _ := os.Getwd()
	candidate := filepath.Base(dir)
	if service.ValidateProject(serviceID, candidate) == nil {
		return candidate
	}
	return ""
}

func guessRepoURI() (url string) {
	rawConfig, err := os.Open("./.git/config")
	if err != nil {
		return
	}
	defer func() { _ = rawConfig.Close() }()

	cfg, err := gitconfig.Parse(rawConfig)
	if err != nil {
		return
	}

	originURI, ok := cfg["remote.origin.url"]
	if ok {
		url = originURI
		return
	}
	return
}

func normalizeAbcService(service string) string {
	service = strings.TrimSpace(service)
	if strings.HasPrefix(service, "http://") || strings.HasPrefix(service, "https://") {
		abcURI := regexp.MustCompile(`^https?://abc\.yandex-team\.ru/services/(\w+)(?:/|\?|$)`)
		matches := abcURI.FindStringSubmatch(service)
		if len(matches) == 2 {
			return matches[1]
		}
		return ""
	}

	return service
}

func findInitTargets() []string {
	dir, _ := filepath.Abs(".")
	rawTargets := cli.FindTargets(dir)
	fmt.Printf("Founded %d targets\n\n", len(rawTargets))
	targets := make([]string, len(rawTargets))
	for i, target := range rawTargets {
		targets[i], _ = filepath.Rel(dir, target)
	}
	return targets
}
