package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"strings"
	"time"

	"github.com/spf13/cobra"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi/internal/cli"
	"a.yandex-team.ru/security/yadi/yadi/internal/config"
)

var (
	projectFiles = []string{
		"pom.xml",
		"go.mod",
		"package.json",
		"req*.txt",
		"req*.pip",
		"requirements/*.txt",
		"requirements/*.pip",
		"maven_install.json",
	}

	opts struct {
		Targets         []string
		ProjectService  string
		ProjectName     string
		SeverityLevel   int
		Format          string
		FeedURI         string
		Token           string
		RawSkipIssues   string
		SkipIssues      []string
		ExitStatus      int
		JsOnly          bool
		PythonOnly      bool
		GoOnly          bool
		JavaOnly        bool
		BazelMavenOnly  bool
		Recursive       bool
		DevDependencies bool
		Local           bool
		Remote          bool
		NoSuggest       bool
		Verbose         bool
		NoPypiCache     bool
		SkipVersion     bool
		ShowVersion     bool
	}
)

// RootCmd represents the base command when called without any subcommands
var RootCmd = &cobra.Command{
	Use:          "yadi",
	SilenceUsage: true,
	Short:        "Yandex Dependency Inspector",
	Long: `Yandex Dependency Inspector

Home page: https://yadi.yandex-team.ru/
Documentation: https://wiki.yandex-team.ru/product-security/yadi/
Vulnerability Database: https://yadi.yandex-team.ru/vulns`,
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		var root *cobra.Command
		for p := cmd; p != nil; p = p.Parent() {
			if p.Name() == "yadi" {
				root = p
				break
			}
		}

		if root == nil {
			return errors.New("can't find root command ('yadi')")
		}

		if err := parseOpts(root.PersistentFlags()); err != nil {
			return err
		}

		if opts.ShowVersion {
			fmt.Println(config.FullVersion())
			os.Exit(0)
		}

		if err := flagsToConfig(); err != nil {
			return err
		}

		if opts.Verbose {
			simplelog.SetLevel(simplelog.DebugLevel)
		} else {
			simplelog.SetLevel(simplelog.InfoLevel)
		}

		if !opts.SkipVersion {
			checkVersion()
		}

		return nil
	},
}

// Execute adds all child commands to the root command sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	RootCmd.Version = config.FullVersion()
	RootCmd.SetVersionTemplate("{{.Version}}\n")

	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func init() {
	flags := RootCmd.PersistentFlags()
	flags.BoolVar(&opts.JsOnly, "js", false,
		"use only Node.js projects")
	flags.BoolVar(&opts.PythonOnly, "py", false,
		"use only Python projects")
	flags.BoolVar(&opts.GoOnly, "go", false,
		"use only Golang projects")
	flags.BoolVar(&opts.JavaOnly, "java", false,
		"use only Java projects")
	flags.BoolVar(&opts.BazelMavenOnly, "bazel", false,
		"use only Bazel projects which utilize rules_jvm_external")
	flags.IntVar(&opts.ExitStatus, "exit-status", 3,
		"set specified exit status when something found")
	flags.BoolVarP(&opts.Recursive, "recursive", "r", false,
		"recursive project search")
	_ = flags.Bool("changes", false,
		"check only new dependencies")
	flags.CountVarP(&opts.SeverityLevel, "level", "l",
		"report issues of a given severity level or higher (-l for Low, -ll for Medium, -lll for High, -llll Critical) (default: Medium)")
	flags.StringVarP(&opts.Format, "format", "f", cli.DetectDefaultFormatter(),
		"output format (text, console, json, st or teamcity)")
	flags.StringVarP(&opts.RawSkipIssues, "skips", "s", "",
		"comma separated list of skipped issues")
	flags.BoolVar(&opts.DevDependencies, "dev", false,
		"include dev dependencies (defaults to production only)")
	flags.BoolVar(&opts.Local, "local", false,
		"use local packages to resolve dependencies")
	flags.BoolVar(&opts.Remote, "remote", false,
		"use remote repository to resolve dependencies")
	flags.BoolVar(&opts.NoSuggest, "no-suggest", false,
		"disable direct update suggesting")
	flags.StringVar(&opts.FeedURI, "feed", "",
		"vulnerability DB URI")
	flags.BoolVar(&opts.Verbose, "verbose", false,
		"verbose output")
	flags.BoolVar(&opts.SkipVersion, "skip-version-check", false,
		"skip version check")
	flags.BoolVar(&opts.NoPypiCache, "no-pypi-cache", false,
		"don't use pypi cache")
	flags.BoolVar(&opts.ShowVersion, "version", false,
		"show the current version")
	flags.StringVar(&opts.ProjectName, "project", "", "project name")

	_ = flags.MarkHidden("changes")
}

func flagsToConfig() error {
	score, err := cvs.FromLevel(opts.SeverityLevel)
	if err != nil {
		return err
	}

	config.NoPypiCache = opts.NoPypiCache
	config.SuggestUpdate = !opts.NoSuggest
	config.SkipIssues = opts.SkipIssues
	config.MinimumSeverity = score
	config.CheckDevDependencies = opts.DevDependencies
	if opts.FeedURI != "" {
		config.FeedURI = opts.FeedURI
	}

	if opts.JsOnly && opts.PythonOnly {
		return errors.New("can't use both --js and --py, please choose what do you wan't")
	}

	if opts.JsOnly || opts.PythonOnly || opts.GoOnly || opts.JavaOnly || opts.BazelMavenOnly {
		config.AllowPython = opts.PythonOnly
		config.AllowJs = opts.JsOnly
		config.AllowGo = opts.GoOnly
		config.AllowJava = opts.JavaOnly
		config.AllowBazelMaven = opts.BazelMavenOnly
	}

	if err := cli.CheckFormatter(opts.Format); err != nil {
		return err
	}

	return nil
}

func parseOpts(flags *pflag.FlagSet) error {
	if err := parseProject(flags); err != nil {
		return err
	}
	// Post process skips
	if opts.RawSkipIssues != "" {
		opts.SkipIssues = []string{}
		skips := strings.Split(opts.RawSkipIssues, ",")
		for _, skip := range skips {
			opts.SkipIssues = append(opts.SkipIssues, strings.TrimSpace(skip))
		}
	}

	return nil
}

func parseProject(flags *pflag.FlagSet) error {
	if _, err := os.Stat(".yadi.toml"); os.IsNotExist(err) {
		// If we don't have any project - just skip :)
		return nil
	}

	project, err := cli.ReadConfig(".yadi.toml")
	if err != nil {
		return err
	}

	if project.JsOnly && project.PythonOnly {
		return errors.New("can't use both JsOnly and PythonOnly, please choose what do you wan't")
	}

	if !flags.Changed("project") {
		opts.ProjectName = project.Name
	}
	opts.ProjectService = project.Service
	opts.Targets = project.Targets
	if token := os.Getenv("YADI_TOKEN"); token != "" {
		opts.Token = token
	} else {
		opts.Token = project.Token
	}

	if !flags.Changed("js") {
		opts.JsOnly = project.JsOnly
	}
	if !flags.Changed("py") {
		opts.PythonOnly = project.PythonOnly
	}
	if !flags.Changed("level") {
		opts.SeverityLevel = cli.SeverityToLevel(project.Severity)
	}
	if !flags.Changed("no-suggest") {
		opts.NoSuggest = project.NoSuggest
	}
	if !flags.Changed("dev") {
		opts.DevDependencies = project.IncludeDev
	}
	if !flags.Changed("skips") {
		opts.SkipIssues = project.Skips
	}

	return nil
}

func checkVersion() {
	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
	defer cancel()

	stClient := sectools.NewClient(config.ProductName)
	isLatest, latestVersion, err := stClient.IsLatestVersion(ctx, config.FullVersion())
	if !isLatest && err != nil {
		simplelog.Warn("----------")
		simplelog.Warn(fmt.Sprintf("new version available: %s", latestVersion))
		if howTo := os.Getenv("YADI_UPDATE_HELP"); howTo != "" {
			simplelog.Warn(howTo)
		}
		simplelog.Warn("----------")
	}
}
